package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.directory.MultiValuesQueriesSupport;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsCacheProvider;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsIterator;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsProvider;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsProviderBuilder;
import com.atlassian.crowd.model.NameComparator;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.search.util.ResultsAggregator;
import com.atlassian.crowd.search.util.ResultsAggregators;
import com.atlassian.crowd.search.util.SearchResultsUtil;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.SetMultimap;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Searching infrastructure that properly belongs to {@code RemoteDirectory} but has historically been implemented
 * in {@code DirectoryManagerGeneric}.
 * <p>
 * This has been factored out in preparation for for migrating it to a more sensible home in the future.
 * </p>
 *
 * @since v2.8.0
 */
class RemoteDirectorySearcher {
    private static final int MEMBERSHIP_BATCH_SIZE_FOR_INTERNAL_DIR = 1000;

    private final RemoteDirectory remoteDirectory;
    private final Optional<NestedGroupsCacheProvider> cacheProvider;

    RemoteDirectorySearcher(@Nonnull RemoteDirectory remoteDirectory, Optional<NestedGroupsCacheProvider> cacheProvider) {
        this.remoteDirectory = checkNotNull(remoteDirectory, "remoteDirectory");
        this.cacheProvider = cacheProvider;
    }

    boolean isUserDirectGroupMember(final String userName, final String groupName)
            throws OperationFailedException {
        return remoteDirectory.isUserDirectGroupMember(userName, groupName);
    }

    boolean isGroupDirectGroupMember(final String childGroup, final String parentGroup)
            throws OperationFailedException {
        return !childGroup.equals(parentGroup)
                && remoteDirectory.supportsNestedGroups()
                && remoteDirectory.isGroupDirectGroupMember(childGroup, parentGroup);
    }

    <T> List<T> searchDirectGroupRelationships(final MembershipQuery<T> query)
            throws OperationFailedException {
        if (!isQuerySupported(query)) {
            return ImmutableList.of();
        }
        if (query.getEntityNamesToMatch().size() <= 1 || remoteDirectory instanceof MultiValuesQueriesSupport) {
            return remoteDirectory.searchGroupRelationships(query);
        }
        ResultsAggregator<T> results = ResultsAggregators.with(query);
        for (MembershipQuery<T> subQuery : query.splitEntityNamesToMatch()) {
            results.addAll(remoteDirectory.searchGroupRelationships(subQuery));
        }
        return results.constrainResults();
    }

    <T> ListMultimap<String, T> searchDirectGroupRelationshipsGroupedByName(final MembershipQuery<T> query) throws OperationFailedException {
        if (query.getEntityNamesToMatch().size() > 1) {
            Preconditions.checkArgument(query.getStartIndex() == 0);
            Preconditions.checkArgument(query.getMaxResults() == EntityQuery.ALL_RESULTS);
        }
        if (!isQuerySupported(query)) {
            return ImmutableListMultimap.of();
        }
        if (remoteDirectory instanceof MultiValuesQueriesSupport) {
            return ((MultiValuesQueriesSupport) remoteDirectory).searchGroupRelationshipsGroupedByName(query);
        }
        final ListMultimap<String, T> resultMap = ArrayListMultimap.create();
        for (String entityToMatch : query.getEntityNamesToMatch()) {
            resultMap.putAll(entityToMatch, remoteDirectory.searchGroupRelationships(query.withEntityNames(entityToMatch)));
        }
        return resultMap;
    }

    private <T> boolean isQuerySupported(final MembershipQuery<T> query) {
        if (isNestedGroupQuery(query) && !remoteDirectory.supportsNestedGroups()) {
            return false;
        }
        if (isLegacyQuery(query)) {
            return false;
        }
        return true;
    }

    private static <T> boolean isNestedGroupQuery(MembershipQuery<T> query) {
        return query.getEntityToMatch().getEntityType() == Entity.GROUP
                && query.getEntityToReturn().getEntityType() == Entity.GROUP;
    }

    private static <T> boolean isLegacyQuery(MembershipQuery<T> query) {
        return isLegacyRole(query.getEntityToMatch()) || isLegacyRole(query.getEntityToReturn());
    }

    private static boolean isLegacyRole(EntityDescriptor entity) {
        return entity.getEntityType() == Entity.GROUP && entity.getGroupType() == GroupType.LEGACY_ROLE;
    }

    boolean isUserNestedGroupMember(final String username, final String groupName)
            throws OperationFailedException {
        List<String> directGroups = searchDirectGroupRelationships(
                createUserParentsQuery(String.class, GroupType.GROUP, ImmutableSet.of(username)));
        return !directGroups.isEmpty()
                && allSubGroupsIterator(ImmutableList.of(groupName), GroupType.GROUP)
                .anyMatch(IdentifierUtils.containsIdentifierPredicate(directGroups)::test);
    }

    public Set<String> filterNestedUserMembersOfGroups(final Set<String> userNames, final Set<String> groupNames)
            throws OperationFailedException {
        SetMultimap<String, String> parentsByUsername = findAllDirectParentsOfTheUsers(userNames);
        SetMultimap<String, String> usersByGroupname = HashMultimap.create();
        parentsByUsername.entries().forEach(entry -> usersByGroupname.put(IdentifierUtils.toLowerCase(entry.getValue()), entry.getKey()));
        Set<String> result = new HashSet<>();
        NestedGroupsIterator<String> groupsIterator = allSubGroupsIterator(groupNames, GroupType.GROUP);
        while (!parentsByUsername.isEmpty() && groupsIterator.hasNext()) {
            Set<String> users = usersByGroupname.get(IdentifierUtils.toLowerCase(groupsIterator.next()));
            result.addAll(users);
            parentsByUsername.keySet().removeAll(users);
        }
        return filterNames(userNames, result);
    }

    private Set<String> filterNames(Collection<String> input, Collection<String> filter) {
        return input.stream().filter(IdentifierUtils.containsIdentifierPredicate(filter)).collect(Collectors.toSet());
    }

    private SetMultimap<String,String> findAllDirectParentsOfTheUsers(Set<String> userNames) throws OperationFailedException {
        MembershipQuery<String> query = createUserParentsQuery(String.class, GroupType.GROUP, userNames);
        return HashMultimap.create(searchDirectGroupRelationshipsGroupedByName(query));
    }

    boolean isGroupNestedGroupMember(final String childGroup, final String parentGroup)
            throws OperationFailedException {
        return remoteDirectory.supportsNestedGroups()
                && !IdentifierUtils.equalsInLowerCase(childGroup, parentGroup)
                && allSubGroupsIterator(ImmutableList.of(parentGroup), GroupType.GROUP)
                .anyMatch(g -> IdentifierUtils.equalsInLowerCase(g, childGroup));
    }

    @SuppressWarnings("unchecked")
    <T> List<T> searchNestedGroupRelationships(final MembershipQuery<T> query) throws OperationFailedException {
        if (!remoteDirectory.supportsNestedGroups()) {
            return searchDirectGroupRelationships(query);
        }

        if (query.isFindChildren() && query.getEntityToReturn().getEntityType() == Entity.USER) {
            List<String> allGroups = allSubGroupsIterator(query.getEntityNamesToMatch(), query.getEntityToMatch().getGroupType()).toList();
            return searchDirectGroupRelationships(query.withEntityNames(allGroups));
        }
        if (query.getReturnType() == String.class) {
            List<Group> groups = searchNestedGroups(query.withReturnType(Group.class));
            return (List<T>) SearchResultsUtil.convertEntitiesToNames(groups);
        }
        return (List<T>) searchNestedGroups((MembershipQuery<Group>) query);
    }

    private <T extends Group> List<T> searchNestedGroups(final MembershipQuery<T> query) throws OperationFailedException {
        Preconditions.checkArgument(query.getEntityToReturn().getEntityType() == Entity.GROUP,
                "You can only find the GROUP memberships of USER or GROUP");

        final NestedGroupsIterator<Group> iterator;
        if (query.getEntityToMatch().getEntityType() == Entity.USER) {
            List<T> directParents = searchDirectGroupRelationships(query.withAllResults());
            iterator = NestedGroupsIterator.groupsIterator(directParents, true, provider(query));
        } else {
            iterator = NestedGroupsIterator.groupsIterator(query.getEntityNamesToMatch(), provider(query));
        }
        List<T> allResults = (List<T>) iterator.toList();
        allResults.sort(NameComparator.directoryEntityComparator());
        return SearchResultsUtil.constrainResults(allResults, query.getStartIndex(), query.getMaxResults());
    }

    private <T> MembershipQuery<T> createUserParentsQuery(Class<T> returnedType, GroupType groupType, Set<String> usernames) {
        return QueryBuilder.queryFor(returnedType, EntityDescriptor.group(groupType))
                .parentsOf(EntityDescriptor.user())
                .withNames(usernames)
                .returningAtMost(EntityQuery.ALL_RESULTS);
    }

    private NestedGroupsIterator<String> allSubGroupsIterator(Collection<String> groups, GroupType groupType) {
        return NestedGroupsIterator.namesIterator(groups, true, nestedGroupsProvider(Group.class, true, groupType));
    }

    private NestedGroupsProvider nestedGroupsProvider(Class<? extends Group> cls, boolean isChildrenQuery, GroupType groupType) {
        NestedGroupsProviderBuilder builder = NestedGroupsProviderBuilder.create().useGroupName();
        if (!remoteDirectory.supportsNestedGroups()) {
            return builder.setSingleGroupProvider((name) -> ImmutableList.of()).build();
        }
        builder.setProvider((Collection<String> names) ->
                (ListMultimap<String, Group>) searchDirectGroupRelationshipsGroupedByName(createQuery(cls, isChildrenQuery, groupType, names)))
                .setBatchSize(remoteDirectory instanceof MultiValuesQueriesSupport ? MEMBERSHIP_BATCH_SIZE_FOR_INTERNAL_DIR : 1);
        if (cacheProvider.isPresent() && cls.equals(Group.class)) {
            builder.useCache(cacheProvider.get(), remoteDirectory.getDirectoryId(), isChildrenQuery, groupType);
        }
        return builder.build();
    }

    private MembershipQuery<? extends Group> createQuery(Class<? extends Group> returnClass, boolean isChildrenQuery, GroupType groupType, Collection<String> names) {
        EntityDescriptor groupDescriptor = EntityDescriptor.group(groupType);
        QueryBuilder.PartialEntityQuery<? extends Group> queryBuilder = QueryBuilder.queryFor(returnClass, groupDescriptor);
        return (isChildrenQuery ? queryBuilder.childrenOf(groupDescriptor) : queryBuilder.parentsOf(groupDescriptor))
                .withNames(names)
                .returningAtMost(EntityQuery.ALL_RESULTS);
    }

    private NestedGroupsProvider provider(MembershipQuery<? extends Group> query) {
        Preconditions.checkArgument(Group.class.isAssignableFrom(query.getReturnType()));
        return nestedGroupsProvider(query.getReturnType(), query.isFindChildren(),
                query.isFindChildren() ? query.getEntityToMatch().getGroupType() : query.getEntityToReturn().getGroupType());
    }
}
