package com.atlassian.crowd.manager.directory;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.core.event.MultiEventPublisher;
import com.atlassian.crowd.dao.application.ApplicationDAO;
import com.atlassian.crowd.directory.InternalRemoteDirectory;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.SynchronisableDirectory;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationInformation;
import com.atlassian.crowd.embedded.api.OperationType;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.event.directory.DirectoryCreatedEvent;
import com.atlassian.crowd.event.directory.DirectoryDeletedEvent;
import com.atlassian.crowd.event.directory.DirectoryUpdatedEvent;
import com.atlassian.crowd.event.group.GroupAttributeDeletedEvent;
import com.atlassian.crowd.event.group.GroupAttributeStoredEvent;
import com.atlassian.crowd.event.group.GroupCreatedEvent;
import com.atlassian.crowd.event.group.GroupDeletedEvent;
import com.atlassian.crowd.event.group.GroupMembershipCreatedEvent;
import com.atlassian.crowd.event.group.GroupMembershipDeletedEvent;
import com.atlassian.crowd.event.group.GroupMembershipsCreatedEvent;
import com.atlassian.crowd.event.group.GroupUpdatedEvent;
import com.atlassian.crowd.event.login.AllPasswordsExpiredEvent;
import com.atlassian.crowd.event.user.UserAttributeDeletedEvent;
import com.atlassian.crowd.event.user.UserAttributeStoredEvent;
import com.atlassian.crowd.event.user.UserCreatedEvent;
import com.atlassian.crowd.event.user.UserCredentialUpdatedEvent;
import com.atlassian.crowd.event.user.UserCredentialValidationFailed;
import com.atlassian.crowd.event.user.UserDeletedEvent;
import com.atlassian.crowd.event.user.UserEditedEvent;
import com.atlassian.crowd.event.user.UserEmailChangedEvent;
import com.atlassian.crowd.event.user.UserRenamedEvent;
import com.atlassian.crowd.event.user.UsersDeletedEvent;
import com.atlassian.crowd.exception.DirectoryCurrentlySynchronisingException;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.NestedGroupsNotSupportedException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerManager;
import com.atlassian.crowd.manager.directory.nestedgroups.NestedGroupsCacheProvider;
import com.atlassian.crowd.manager.permission.PermissionManager;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.NameComparator;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.directory.ImmutableDirectory;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.membership.MembershipType;
import com.atlassian.crowd.model.user.ImmutableUser;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.util.BatchResult;
import com.atlassian.crowd.util.BoundedCount;
import com.atlassian.crowd.util.DirectoryEntityUtils;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Nonnull;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.stream.Collectors;

import static com.atlassian.crowd.manager.directory.SynchronisationMode.FULL;
import static com.atlassian.crowd.manager.directory.SynchronisationMode.INCREMENTAL;
import static com.atlassian.crowd.search.query.DirectoryQueries.allDirectories;
import static com.google.common.base.Preconditions.checkNotNull;

@Transactional
public class DirectoryManagerGeneric implements DirectoryManager {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryManagerGeneric.class);

    private final DirectoryDao directoryDao;
    private final ApplicationDAO applicationDAO;
    private final MultiEventPublisher eventPublisher;
    private final PermissionManager permissionManager;
    private final DirectoryInstanceLoader directoryInstanceLoader;
    private final DirectorySynchroniser directorySynchroniser;
    private final DirectoryPollerManager directoryPollerManager;
    private final ClusterLockService lockService;
    private final SynchronisationStatusManager synchronisationStatusManager;
    private final BeforeGroupRemoval beforeGroupRemoval;
    private final Optional<NestedGroupsCacheProvider> nestedGroupsCacheProvider;

    public DirectoryManagerGeneric(DirectoryDao directoryDao,
                                   ApplicationDAO applicationDAO,
                                   MultiEventPublisher eventPublisher,
                                   PermissionManager permissionManager,
                                   DirectoryInstanceLoader directoryInstanceLoader,
                                   DirectorySynchroniser directorySynchroniser,
                                   DirectoryPollerManager directoryPollerManager,
                                   ClusterLockService lockService,
                                   SynchronisationStatusManager synchronisationStatusManager,
                                   BeforeGroupRemoval beforeGroupRemoval,
                                   Optional<NestedGroupsCacheProvider> nestedGroupsCacheProvider) {
        this.directoryDao = checkNotNull(directoryDao);
        this.applicationDAO = checkNotNull(applicationDAO);
        this.eventPublisher = checkNotNull(eventPublisher);
        this.permissionManager = checkNotNull(permissionManager);
        this.directoryInstanceLoader = checkNotNull(directoryInstanceLoader);
        this.directorySynchroniser = checkNotNull(directorySynchroniser);
        this.directoryPollerManager = checkNotNull(directoryPollerManager);
        this.lockService = checkNotNull(lockService);
        this.synchronisationStatusManager = checkNotNull(synchronisationStatusManager);
        this.beforeGroupRemoval = checkNotNull(beforeGroupRemoval);
        this.nestedGroupsCacheProvider = nestedGroupsCacheProvider;
    }

    @Override
    public Directory addDirectory(final Directory directory) throws DirectoryInstantiationException {
        // make sure the raw implementation is instantiable
        if (!directoryInstanceLoader.canLoad(directory.getImplementationClass())) {
            throw new IllegalArgumentException("Failed to instantiate directory with class: " + directory.getImplementationClass());
        }

        Directory addedDirectory = directoryDao.add(directory);

        eventPublisher.publish(new DirectoryCreatedEvent(this, addedDirectory));

        return addedDirectory;
    }

    @Override
    public Directory findDirectoryById(final long directoryId) throws DirectoryNotFoundException {
        return directoryDao.findById(directoryId);
    }

    @Override
    public List<Directory> findAllDirectories() {
        return directoryDao.search(allDirectories());
    }

    @Override
    public List<Directory> searchDirectories(final EntityQuery<Directory> query) {
        return directoryDao.search(query);
    }

    @Override
    public Directory findDirectoryByName(final String name) throws DirectoryNotFoundException {
        return directoryDao.findByName(name);
    }

    @Override
    public Directory updateDirectory(final Directory directory) throws DirectoryNotFoundException {
        if (directory.getId() == null) {
            throw new DirectoryNotFoundException(directory.getId());
        }

        // throws DNFE if directory does not exist
        final Directory oldDirectory = ImmutableDirectory.from(findDirectoryById(directory.getId()));

        Directory updatedDirectory = directoryDao.update(directory);

        eventPublisher.publish(new DirectoryUpdatedEvent(this, oldDirectory, updatedDirectory));

        return updatedDirectory;
    }

    @Override
    public void removeDirectory(final Directory directory) throws DirectoryNotFoundException, DirectoryCurrentlySynchronisingException {
        Lock lock = lockService.getLockForName(DirectorySynchronisationUtils.getLockName(directory.getId()));
        if (lock.tryLock()) {
            try {
                // remove all application associations
                applicationDAO.removeDirectoryMappings(directory.getId());

                // remove all statuses for directory
                synchronisationStatusManager.removeStatusesForDirectory(directory.getId());

                synchronisationStatusManager.clearSynchronisationTokenForDirectory(directory.getId());

                // remove the directory (and associated internal users/groups/memberships)
                directoryDao.remove(directory);

                eventPublisher.publish(new DirectoryDeletedEvent(this, directory));
            } finally {
                lock.unlock();
            }
        } else {
            // unfortunately, due to the way we currently check if a directory is synchronising (i.e. directory is
            // synchronising if we can't acquire the lock), if another thread successfully acquired the lock while
            // checking if the directory is synchronising, this thread will falsely throw a
            // {@link DirectoryCurrentlySynchronisingException}
            throw new DirectoryCurrentlySynchronisingException(directory.getId());
        }
    }

    @Override
    public boolean supportsNestedGroups(final long directoryId) throws DirectoryInstantiationException, DirectoryNotFoundException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        return remoteDirectory.supportsNestedGroups();
    }

    @Override
    public boolean isSynchronisable(final long directoryId) throws DirectoryInstantiationException, DirectoryNotFoundException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        return isSynchronisable(remoteDirectory);
    }

    private static boolean isSynchronisable(RemoteDirectory remoteDirectory) {
        return remoteDirectory instanceof SynchronisableDirectory;
    }

    @Override
    public SynchronisationMode getSynchronisationMode(long directoryId) throws DirectoryInstantiationException, DirectoryNotFoundException {
        final RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        return isSynchronisable(remoteDirectory) ? getSynchronisationMode((SynchronisableDirectory) remoteDirectory) : null;
    }

    private static SynchronisationMode getSynchronisationMode(SynchronisableDirectory directory) {
        return directory.isIncrementalSyncEnabled() ? INCREMENTAL : FULL;
    }

    @Override
    public void synchroniseCache(final long directoryId, final SynchronisationMode mode)
            throws OperationFailedException, DirectoryNotFoundException {
        synchroniseCache(directoryId, mode, true);
    }

    @Override
    public void synchroniseCache(final long directoryId, final SynchronisationMode mode, final boolean runInBackground)
            throws OperationFailedException, DirectoryNotFoundException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        if (isSynchronisable(remoteDirectory)) {
            if (runInBackground) {
                directoryPollerManager.triggerPoll(directoryId, mode);
            } else {
                if (isSynchronising(directoryId)) {
                    throw new OperationFailedException("Directory " + directoryId + " is currently synchronising");
                }
                directorySynchroniser.synchronise((SynchronisableDirectory) remoteDirectory, mode);
            }
        }
    }

    @Override
    public boolean isSynchronising(final long directoryId)
            throws DirectoryInstantiationException, DirectoryNotFoundException {
        return directorySynchroniser.isSynchronising(directoryId);
    }

    @Override
    public DirectorySynchronisationInformation getDirectorySynchronisationInformation(long directoryId)
            throws DirectoryInstantiationException, DirectoryNotFoundException {
        final RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        if (isSynchronisable(remoteDirectory)) {
            return synchronisationStatusManager.getDirectorySynchronisationInformation(directoryId);
        }
        return null;
    }

    /////////// USER OPERATIONS ///////////

    private RemoteDirectory getDirectoryImplementation(long directoryId) throws DirectoryInstantiationException, DirectoryNotFoundException {
        return directoryInstanceLoader.getDirectory(findDirectoryById(directoryId));
    }

    private RemoteDirectorySearcher getSearcher(long directoryId) throws DirectoryInstantiationException, DirectoryNotFoundException {
        return new RemoteDirectorySearcher(getDirectoryImplementation(directoryId), nestedGroupsCacheProvider);
    }

    @Override
    public User authenticateUser(long directoryId, String username, PasswordCredential passwordCredential)
            throws OperationFailedException, InactiveAccountException, InvalidAuthenticationException, ExpiredCredentialException, DirectoryNotFoundException, UserNotFoundException {
        return getDirectoryImplementation(directoryId).authenticate(username, passwordCredential);
    }

    @Override
    public User userAuthenticated(long directoryId, String username) throws OperationFailedException, DirectoryNotFoundException, UserNotFoundException, InactiveAccountException {
        return getDirectoryImplementation(directoryId).userAuthenticated(username);
    }

    @Override
    public User findUserByName(final long directoryId, final String username)
            throws OperationFailedException, DirectoryNotFoundException, UserNotFoundException {
        return getDirectoryImplementation(directoryId).findUserByName(username);
    }

    @Override
    public UserWithAttributes findUserWithAttributesByName(final long directoryId, final String username)
            throws OperationFailedException, DirectoryNotFoundException, UserNotFoundException {
        return getDirectoryImplementation(directoryId).findUserWithAttributesByName(username);
    }

    @Override
    public <T> List<T> searchUsers(final long directoryId, final EntityQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        return getDirectoryImplementation(directoryId).searchUsers(query);
    }

    @Override
    public User addUser(final long directoryId, final UserTemplate user, final PasswordCredential credential)
            throws InvalidCredentialException, InvalidUserException, OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException, UserAlreadyExistsException {
        return addUser(directoryId, UserTemplateWithAttributes.toUserWithNoAttributes(user), credential);
    }

    @Override
    public UserWithAttributes addUser(final long directoryId, final UserTemplateWithAttributes user, final PasswordCredential credential)
            throws InvalidCredentialException, InvalidUserException, OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException, UserAlreadyExistsException {
        if (userExists(directoryId, user.getName())) {
            throw new UserAlreadyExistsException(directoryId, user.getName());
        }

        if (IdentifierUtils.hasLeadingOrTrailingWhitespace(user.getName())) {
            throw new InvalidUserException(user, "User name may not contain leading or trailing whitespace");
        }

        Directory directory = findDirectoryById(directoryId);

        if (permissionManager.hasPermission(directory, OperationType.CREATE_USER)) {
            UserWithAttributes createdUser = getDirectoryImplementation(directoryId).addUser(user, credential);

            eventPublisher.publish(new UserCreatedEvent(this, directory, createdUser));

            return createdUser;
        } else {
            throw new DirectoryPermissionException("Directory does not allow adding of users");
        }
    }

    /**
     * Returns true if the user exists.
     *
     * @param directoryId directory ID
     * @param username    name of the user
     * @return true if the user exists, otherwise false.
     * @throws DirectoryNotFoundException if the directory could not be found
     * @throws OperationFailedException   if the operation failed for any other reason
     */
    private boolean userExists(final long directoryId, final String username)
            throws DirectoryNotFoundException, OperationFailedException {
        try {
            findUserByName(directoryId, username);
            return true;
        } catch (UserNotFoundException e) {
            return false;
        }
    }

    @Override
    public User updateUser(final long directoryId, final UserTemplate user)
            throws OperationFailedException, DirectoryPermissionException, InvalidUserException, DirectoryNotFoundException, UserNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        if (permissionManager.hasPermission(directory, OperationType.UPDATE_USER)) {
            final RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
            User currentUser = ImmutableUser.from(remoteDirectory.findUserByName(user.getName()));
            User updatedUser = remoteDirectory.updateUser(user);

            eventPublisher.publish(new UserEditedEvent(this, directory, updatedUser, currentUser));

            String originalEmail = currentUser.getEmailAddress();
            if (!Objects.equals(originalEmail, updatedUser.getEmailAddress())) {
                eventPublisher.publish(new UserEmailChangedEvent(this, directory, updatedUser, originalEmail));
            }

            return updatedUser;
        } else {
            throw new DirectoryPermissionException("Directory does not allow user modifications");
        }
    }

    @Override
    public User renameUser(final long directoryId, final String oldUsername, final String newUsername)
            throws OperationFailedException, DirectoryPermissionException, InvalidUserException, DirectoryNotFoundException, UserNotFoundException, UserAlreadyExistsException {
        if (IdentifierUtils.hasLeadingOrTrailingWhitespace(newUsername)) {
            com.atlassian.crowd.embedded.api.User user = new UserTemplate(newUsername, directoryId);
            throw new InvalidUserException(user, "User name may not contain leading or trailing whitespace");
        }

        Directory directory = findDirectoryById(directoryId);
        if (permissionManager.hasPermission(directory, OperationType.UPDATE_USER)) {
            final RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
            if (newUsername.equals(oldUsername)) { // trivial rename, case sensitive comparison
                return remoteDirectory.findUserByName(oldUsername);
            } else {
                User updatedUser = remoteDirectory.renameUser(oldUsername, newUsername);

                eventPublisher.publish(new UserRenamedEvent(this, directory, updatedUser, oldUsername));

                return updatedUser;
            }
        } else {
            throw new DirectoryPermissionException("Directory does not allow user modifications");
        }
    }

    @Override
    public void storeUserAttributes(final long directoryId, final String username, final Map<String, Set<String>> attributes)
            throws OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException, UserNotFoundException {
        Directory directory = findDirectoryById(directoryId);
        if (permissionManager.hasPermission(directory, OperationType.UPDATE_USER_ATTRIBUTE)) {
            getDirectoryImplementation(directoryId).storeUserAttributes(username, attributes);

            User updatedUser = getDirectoryImplementation(directoryId).findUserByName(username);

            eventPublisher.publish(new UserAttributeStoredEvent(this, directory, updatedUser, attributes));
        } else {
            throw new DirectoryPermissionException("Directory does not allow user attribute modifications");
        }
    }

    @Override
    public void removeUserAttributes(final long directoryId, final String username, final String attributeName)
            throws OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException, UserNotFoundException {
        Directory directory = findDirectoryById(directoryId);
        if (permissionManager.hasPermission(directory, OperationType.UPDATE_USER_ATTRIBUTE)) {
            getDirectoryImplementation(directoryId).removeUserAttributes(username, attributeName);
            User updatedUser = getDirectoryImplementation(directoryId).findUserByName(username);

            eventPublisher.publish(new UserAttributeDeletedEvent(this, directory, updatedUser, attributeName));
        } else {
            throw new DirectoryPermissionException("Directory does not allow user attribute modifications");
        }
    }

    @Override
    public void updateUserCredential(final long directoryId, final String username, final PasswordCredential credential)
            throws OperationFailedException, DirectoryPermissionException, InvalidCredentialException, DirectoryNotFoundException, UserNotFoundException {
        Directory directory = findDirectoryById(directoryId);
        if (permissionManager.hasPermission(directory, OperationType.UPDATE_USER)) {
            logger.info("The password for \"" + username + "\" in \"" + directory.getName() + "\" is being changed.");

            try {
                getDirectoryImplementation(directoryId).updateUserCredential(username, credential);
                eventPublisher.publish(new UserCredentialUpdatedEvent(this, directory, username, credential));
            } catch (InvalidCredentialException invalidCredentialException) {
                eventPublisher.publish(new UserCredentialValidationFailed(this, directory, invalidCredentialException.getViolatedConstraints()));
                throw invalidCredentialException;
            }
        } else {
            throw new DirectoryPermissionException("Directory does not allow user modifications");
        }
    }

    @Override
    public void removeUser(final long directoryId, final String username)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException, UserNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        if (permissionManager.hasPermission(directory, OperationType.DELETE_USER)) {
            getDirectoryImplementation(directoryId).removeUser(username);

            eventPublisher.publish(new UsersDeletedEvent(this, directory, Collections.singleton(username)));
            // legacy event
            eventPublisher.publish(new UserDeletedEvent(this, directory, username));
        } else {
            throw new DirectoryPermissionException("Directory does not allow user removal");
        }
    }

    /////////// GROUP OPERATIONS ///////////

    @Override
    public Group findGroupByName(final long directoryId, final String groupName)
            throws OperationFailedException, GroupNotFoundException, DirectoryNotFoundException {
        return getDirectoryImplementation(directoryId).findGroupByName(groupName);
    }

    @Override
    public GroupWithAttributes findGroupWithAttributesByName(final long directoryId, final String groupName)
            throws OperationFailedException, GroupNotFoundException, DirectoryNotFoundException {
        return getDirectoryImplementation(directoryId).findGroupWithAttributesByName(groupName);
    }

    @Override
    public <T> List<T> searchGroups(final long directoryId, final EntityQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        return getDirectoryImplementation(directoryId).searchGroups(query);
    }

    @Override
    public Group addGroup(final long directoryId, final GroupTemplate group)
            throws InvalidGroupException, OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException {
        Group createdGroup;
        Directory directory = findDirectoryById(directoryId);
        try {
            findGroupByName(directoryId, group.getName());
            throw new InvalidGroupException(group, "Group with name <" + group.getName() + "> already exists in directory <" + directory.getName() + ">");
        } catch (GroupNotFoundException e) {
            if (IdentifierUtils.hasLeadingOrTrailingWhitespace(group.getName())) {
                throw new InvalidGroupException(group, "Group name may not contain leading or trailing whitespace");
            }

            final OperationType operationType = getCreateOperationType(group);

            // only add a group if we can't find it
            if (permissionManager.hasPermission(directory, operationType)) {
                createdGroup = getDirectoryImplementation(directoryId).addGroup(group);

                eventPublisher.publish(new GroupCreatedEvent(this, directory, createdGroup));
            } else {
                if (operationType.equals(OperationType.CREATE_GROUP)) {
                    throw new DirectoryPermissionException("Directory does not allow adding of groups");
                } else {
                    throw new DirectoryPermissionException("Directory does not allow adding of roles");
                }
            }
        }
        return createdGroup;
    }

    @Override
    public Group updateGroup(final long directoryId, final GroupTemplate group)
            throws OperationFailedException, DirectoryPermissionException, InvalidGroupException, DirectoryNotFoundException, GroupNotFoundException, ReadOnlyGroupException {
        Group updatedGroup;
        Directory directory = findDirectoryById(directoryId);

        final OperationType operationType = getUpdateOperationType(group);

        if (permissionManager.hasPermission(directory, operationType)) {
            updatedGroup = getDirectoryImplementation(directoryId).updateGroup(group);

            eventPublisher.publish(new GroupUpdatedEvent(this, directory, updatedGroup));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }

        }

        return updatedGroup;
    }

    @Override
    public Group renameGroup(final long directoryId, final String oldGroupname, final String newGroupname)
            throws OperationFailedException, DirectoryPermissionException, InvalidGroupException, DirectoryNotFoundException, GroupNotFoundException {
        if (IdentifierUtils.hasLeadingOrTrailingWhitespace(newGroupname)) {
            Group group = new GroupTemplate(newGroupname, directoryId);
            throw new InvalidGroupException(group, "Group name may not contain leading or trailing whitespace");
        }

        Directory directory = findDirectoryById(directoryId);

        final Group groupToUpdate = findGroupByName(directoryId, oldGroupname);
        final OperationType operationType = getUpdateOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            final Group updatedGroup = getDirectoryImplementation(directoryId).renameGroup(oldGroupname, newGroupname);

            eventPublisher.publish(new GroupUpdatedEvent(this, directory, updatedGroup));

            return updatedGroup;
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    @Override
    public void storeGroupAttributes(final long directoryId, final String groupName, final Map<String, Set<String>> attributes)
            throws OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException, GroupNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        final Group groupToUpdate = findGroupByName(directoryId, groupName);
        final OperationType operationType = getUpdateAttributeOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            getDirectoryImplementation(directoryId).storeGroupAttributes(groupName, attributes);

            final Group updateGroup = findGroupByName(directoryId, groupName);

            eventPublisher.publish(new GroupAttributeStoredEvent(this, directory, updateGroup, attributes));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    @Override
    public void removeGroupAttributes(final long directoryId, final String groupName, final String attributeName)
            throws OperationFailedException, DirectoryPermissionException, DirectoryNotFoundException, GroupNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        final Group groupToUpdate = findGroupByName(directoryId, groupName);
        final OperationType operationType = getUpdateAttributeOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            getDirectoryImplementation(directoryId).removeGroupAttributes(groupName, attributeName);

            final Group updateGroup = findGroupByName(directoryId, groupName);

            eventPublisher.publish(new GroupAttributeDeletedEvent(this, directory, updateGroup, attributeName));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP_ATTRIBUTE)) {
                throw new DirectoryPermissionException("Directory does not allow group attribute modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role attribute modifications");
            }
        }
    }

    @Override
    public void removeGroup(final long directoryId, final String groupName)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException, GroupNotFoundException, ReadOnlyGroupException {
        Directory directory = findDirectoryById(directoryId);

        final Group groupToDelete = findGroupByName(directoryId, groupName);
        final OperationType operationType = getDeleteOperationType(groupToDelete);

        if (permissionManager.hasPermission(directory, operationType)) {
            beforeGroupRemoval.beforeRemoveGroup(directoryId, groupName);
            applicationDAO.removeGroupMappings(directoryId, groupName);

            // remove the group from the underlying directory implementation
            getDirectoryImplementation(directoryId).removeGroup(groupName);

            eventPublisher.publish(new GroupDeletedEvent(this, directory, groupName));
        } else {
            if (operationType.equals(OperationType.DELETE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group removal");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role removal");
            }
        }
    }

    @Override
    public boolean isUserDirectGroupMember(final long directoryId, final String username, final String groupName)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).isUserDirectGroupMember(username, groupName);
    }

    @Override
    public boolean isGroupDirectGroupMember(final long directoryId, final String childGroup, final String parentGroup)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).isGroupDirectGroupMember(childGroup, parentGroup);
    }

    @Override
    public void addUserToGroup(final long directoryId, final String username, final String groupName)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException,
            GroupNotFoundException, UserNotFoundException, ReadOnlyGroupException, MembershipAlreadyExistsException {
        Directory directory = findDirectoryById(directoryId);

        final Group groupToUpdate = findGroupByName(directoryId, groupName);
        final OperationType operationType = getUpdateOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            getDirectoryImplementation(directoryId).addUserToGroup(username, groupName);

            eventPublisher.publish(new GroupMembershipCreatedEvent(this, directory, username, groupName, MembershipType.GROUP_USER));
            eventPublisher.publish(new GroupMembershipsCreatedEvent(this, directory, ImmutableList.of(username), groupName, MembershipType.GROUP_USER));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    @Override
    public void addGroupToGroup(final long directoryId, final String childGroup, final String parentGroup)
            throws DirectoryPermissionException, OperationFailedException, InvalidMembershipException,
            NestedGroupsNotSupportedException, DirectoryNotFoundException, GroupNotFoundException,
            ReadOnlyGroupException, MembershipAlreadyExistsException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        if (!remoteDirectory.supportsNestedGroups()) {
            throw new NestedGroupsNotSupportedException(directoryId);
        }

        Directory directory = findDirectoryById(directoryId);

        final Group parentGroupToUpdate = findGroupByName(directoryId, parentGroup);
        final OperationType operationType = getUpdateOperationType(parentGroupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            if (childGroup.equals(parentGroup)) {
                throw new InvalidMembershipException("Cannot add direct circular group membership reference");
            }

            remoteDirectory.addGroupToGroup(childGroup, parentGroup);

            eventPublisher.publish(new GroupMembershipCreatedEvent(this, directory, childGroup, parentGroup, MembershipType.GROUP_GROUP));
            eventPublisher.publish(new GroupMembershipsCreatedEvent(this, directory, ImmutableList.of(childGroup), parentGroup, MembershipType.GROUP_GROUP));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    @Override
    public void removeUserFromGroup(final long directoryId, final String username, final String groupName)
            throws DirectoryPermissionException, OperationFailedException, MembershipNotFoundException, DirectoryNotFoundException, GroupNotFoundException, UserNotFoundException, ReadOnlyGroupException {
        Directory directory = findDirectoryById(directoryId);

        final Group groupToUpdate = findGroupByName(directoryId, groupName);
        final OperationType operationType = getUpdateOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            getDirectoryImplementation(directoryId).removeUserFromGroup(username, groupName);

            eventPublisher.publish(new GroupMembershipDeletedEvent(this, directory, username, groupName, MembershipType.GROUP_USER));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    @Override
    public void removeGroupFromGroup(final long directoryId, final String childGroup, final String parentGroup)
            throws DirectoryPermissionException, OperationFailedException, InvalidMembershipException, MembershipNotFoundException, DirectoryNotFoundException, GroupNotFoundException, ReadOnlyGroupException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        if (!remoteDirectory.supportsNestedGroups()) {
            throw new UnsupportedOperationException("Directory with id [" + directoryId + "] does not support nested groups");
        }

        Directory directory = findDirectoryById(directoryId);

        final Group groupToUpdate = findGroupByName(directoryId, parentGroup);
        final OperationType operationType = getUpdateOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            if (childGroup.equals(parentGroup)) {
                throw new InvalidMembershipException("Cannot remove direct circular group membership reference");
            }

            remoteDirectory.removeGroupFromGroup(childGroup, parentGroup);

            eventPublisher.publish(new GroupMembershipDeletedEvent(this, directory, childGroup, parentGroup, MembershipType.GROUP_GROUP));
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    @Override
    public <T> List<T> searchDirectGroupRelationships(final long directoryId, final MembershipQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).searchDirectGroupRelationships(query);
    }

    @Override
    public <T> ListMultimap<String, T> searchDirectGroupRelationshipsGroupedByName(final long directoryId, final MembershipQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).searchDirectGroupRelationshipsGroupedByName(query);
    }

    @Override
    public BoundedCount countDirectMembersOfGroup(long directoryId, String groupName, int querySizeHint)
            throws OperationFailedException, DirectoryNotFoundException {
        return getDirectoryImplementation(directoryId).countDirectMembersOfGroup(groupName, querySizeHint);
    }

    @Override
    @Transactional(readOnly = true)
    public boolean isUserNestedGroupMember(final long directoryId, final String username, final String groupName)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).isUserNestedGroupMember(username, groupName);
    }

    @Override
    @Transactional(readOnly = true)
    public boolean isUserNestedGroupMember(final long directoryId, final String username, final Set<String> groupNames)
            throws OperationFailedException, DirectoryNotFoundException {
        return !getSearcher(directoryId).filterNestedUserMembersOfGroups(ImmutableSet.of(username), groupNames).isEmpty();
    }

    @Override
    @Transactional(readOnly = true)
    public Set<String> filterNestedUserMembersOfGroups(final long directoryId, final Set<String> userNames, final Set<String> groupNames)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).filterNestedUserMembersOfGroups(userNames, groupNames);
    }

    @Override
    @Transactional(readOnly = true)
    public boolean isGroupNestedGroupMember(long directoryId, String childGroupName, String parentGroupName)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).isGroupNestedGroupMember(childGroupName, parentGroupName);
    }

    @Override
    @Transactional(readOnly = true)
    public <T> List<T> searchNestedGroupRelationships(final long directoryId, final MembershipQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        return getSearcher(directoryId).searchNestedGroupRelationships(query);
    }


    /////////////// BULK OPERATIONS ///////////////

    @Override
    public BulkAddResult<User> addAllUsers(final long directoryId, final Collection<UserTemplateWithCredentialAndAttributes> users, final boolean overwrite)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        if (!permissionManager.hasPermission(directory, OperationType.CREATE_USER)) {
            throw new DirectoryPermissionException("Directory does not allow adding of users");
        }

        final BulkAddResult.Builder<User> bulkAddResultBuilder = BulkAddResult.<User>builder(users.size())
                .setOverwrite(overwrite);
        final Collection<UserTemplateWithCredentialAndAttributes> usersToAdd;

        // find existing users with the same names and delete them if we're overwriting
        final BulkRemoveResult<String> bulkRemoveExistingResult = removeAllUsers(directoryId, users, overwrite);
        if (overwrite) {
            usersToAdd = Lists.newArrayList(getDeletedAndNotFoundEntities(users, bulkRemoveExistingResult,
                    NameComparator.normaliserOf(UserTemplateWithCredentialAndAttributes.class)));
        } else {
            usersToAdd = Lists.newArrayList(getNotFoundEntities(users, bulkRemoveExistingResult,
                    NameComparator.normaliserOf(UserTemplateWithCredentialAndAttributes.class)));
        }

        // after the remove, any failures -> those users are still there and can't be overwritten
        for (final String stillExistingUserName : bulkRemoveExistingResult.getFailedEntities()) {
            try {
                bulkAddResultBuilder.addExistingEntity(findUserByName(directoryId, stillExistingUserName));
                if (overwrite) {
                    logger.error("User could not be removed for bulk add with overwrite: {}", stillExistingUserName);
                } else {
                    logger.info("User already exists in directory; overwrite is off so import will skip it: {}",
                            stillExistingUserName);
                }
            } catch (UserNotFoundException e) {
                logger.debug("User previously existed and couldn't be removed but is now missing so importing it: {}",
                        stillExistingUserName);
                usersToAdd.add(Iterables.find(users, DirectoryEntityUtils.whereNameEquals(stillExistingUserName)));
            }
        }

        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);

        // retain unique entities
        Set<UserTemplateWithCredentialAndAttributes> uniqueUsersToAdd = retainUniqueEntities(usersToAdd);

        // perform the add operation
        final Collection<User> successfulEntities;
        final Collection<User> failedEntities;
        if (remoteDirectory instanceof InternalRemoteDirectory) {
            final BatchResult<User> batchResult = ((InternalRemoteDirectory) remoteDirectory).addAllUsers(uniqueUsersToAdd);
            successfulEntities = batchResult.getSuccessfulEntities();
            failedEntities = batchResult.getFailedEntities();
        } else {
            successfulEntities = new ArrayList<User>(uniqueUsersToAdd);
            failedEntities = new ArrayList<User>();
            for (UserTemplateWithCredentialAndAttributes user : uniqueUsersToAdd) {
                try {
                    if (IdentifierUtils.hasLeadingOrTrailingWhitespace(user.getName())) {
                        throw new InvalidUserException(user, "User name may not contain leading or trailing whitespace");
                    }

                    successfulEntities.add(remoteDirectory.addUser(user, user.getCredential()));
                } catch (Exception e) {
                    logger.error("Failed to add user: {}", user, e);
                    failedEntities.add(user);
                }
            }
        }

        bulkAddResultBuilder.addFailedEntities(failedEntities);

        logFailedEntities(remoteDirectory, failedEntities);

        // Fire events
        for (User addedUser : successfulEntities) {
            eventPublisher.publish(new UserCreatedEvent(this, directory, addedUser));
        }

        return bulkAddResultBuilder.build();
    }

    private static <T extends DirectoryEntity> Iterable<T> getDeletedAndNotFoundEntities(Collection<T> users,
                                                                                         final BulkRemoveResult<String> bulkRemoveExistingResult,
                                                                                         final Function<T, String> normaliser) {
        return Iterables.filter(users, Predicates.compose(new Predicate<String>() {
            @Override
            public boolean apply(String normalisedUserName) {
                return !bulkRemoveExistingResult.getFailedEntities().contains(normalisedUserName)
                        || bulkRemoveExistingResult.getMissingEntities().contains(normalisedUserName);
            }
        }, normaliser));
    }

    private static <T extends DirectoryEntity> Iterable<T> getNotFoundEntities(Collection<T> users,
                                                                               final BulkRemoveResult<String> bulkRemoveExistingResult,
                                                                               final Function<T, String> normaliser) {
        return Iterables.filter(users, new Predicate<T>() {
            @Override
            public boolean apply(T user) {
                return bulkRemoveExistingResult.getMissingEntities().contains(normaliser.apply(user));
            }
        });
    }

    private <T> Set<T> retainUniqueEntities(final Iterable<T> entities) {
        Set<T> uniqueEntities = new HashSet<T>();
        for (T entity : entities) {
            if (logger.isDebugEnabled()) {
                logger.debug("Going to add: " + entity);
            }

            boolean added = uniqueEntities.add(entity);

            if (!added) {
                logger.warn("Duplicate entity. Entity is already in the set of entities to bulk add: " + entity);
            }
        }
        return uniqueEntities;
    }

    /**
     * Find and optionally (if {@code doRemove} is {@code true}) remove all of the given users from the given directory.
     *
     * @param directoryId id of directory to remove users from
     * @param users       users to find and possibly remove
     * @param doRemove    true to remove users, false to just find them
     * @return a BulkRemoveResult with missingEntities containing those users which were not found, and
     * failedEntities containing either the remainder (when doRemove is false) or those users which were attempted to
     * be removed but could not be (when doRemove is true). In all cases, users are recorded as their names which
     * will have been normalised according to the {@link NameComparator#normaliserOf(Class)} for {@link User}.
     * @throws DirectoryPermissionException if directory does not have {@link OperationType#DELETE_USER} permission
     * @throws OperationFailedException     if the bulk delete failed
     * @throws DirectoryNotFoundException   if the directory with the given id could not be found
     */
    private BulkRemoveResult<String> removeAllUsers(final long directoryId,
                                                    final Collection<? extends UserTemplate> users, boolean doRemove)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        if (!permissionManager.hasPermission(directory, OperationType.DELETE_USER) && doRemove) {
            throw new DirectoryPermissionException("Directory does not allow removing users");
        }

        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);

        List<UserTemplate> usersToRemove = new ArrayList<UserTemplate>();
        BulkRemoveResult.Builder<String> resultBuilder = BulkRemoveResult.builder(users.size());

        final Function<User, String> usernameNormaliser = NameComparator.normaliserOf(User.class);
        for (UserTemplate user : users) {
            try {
                findUserByName(directoryId, usernameNormaliser.apply(user));
                usersToRemove.add(user);
            } catch (UserNotFoundException e) {
                resultBuilder.addMissingEntity(usernameNormaliser.apply(user));
            }
        }

        // retain unique entities
        Set<String> namesOfUniqueUsersToRemove = retainUniqueEntities(
                Iterables.transform(usersToRemove, usernameNormaliser));

        if (!doRemove) {
            // all entities we would remove normally are instead "failed"
            resultBuilder.addFailedEntities(namesOfUniqueUsersToRemove);
        } else {
            // perform the remove operation
            final Collection<String> successfulEntities;
            final Collection<String> failedEntities;
            if (remoteDirectory instanceof InternalRemoteDirectory) {
                BatchResult<String> batchResult = ((InternalRemoteDirectory) remoteDirectory).removeAllUsers(
                        namesOfUniqueUsersToRemove);
                successfulEntities = batchResult.getSuccessfulEntities();
                failedEntities = batchResult.getFailedEntities();
            } else {
                successfulEntities = new ArrayList<String>(namesOfUniqueUsersToRemove.size());
                failedEntities = new ArrayList<String>();
                for (String username : namesOfUniqueUsersToRemove) {
                    try {
                        remoteDirectory.removeUser(username);
                        successfulEntities.add(username);
                    } catch (Exception e) {
                        logger.error("Failed to remove user: {}", username, e);
                        failedEntities.add(username);
                    }
                }
            }

            resultBuilder.addFailedEntities(failedEntities);
            logFailedEntitiesWithNames(remoteDirectory, failedEntities);

            eventPublisher.publish(new UsersDeletedEvent(this, directory, Collections.unmodifiableCollection(successfulEntities)));
            // legacy event
            eventPublisher.publishAll(successfulEntities.stream()
                    .map(username -> new UserDeletedEvent(this, directory, username))
                    .collect(Collectors.toList()));
        }

        return resultBuilder.build();
    }

    @Override
    public BulkAddResult<Group> addAllGroups(final long directoryId, final Collection<GroupTemplate> groups, final boolean overwrite)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        if (!permissionManager.hasPermission(directory, OperationType.CREATE_GROUP)) {
            throw new DirectoryPermissionException("Directory does not allow adding of groups");
        }

        final BulkAddResult.Builder<Group> bulkAddResultBuilder = BulkAddResult.<Group>builder(groups.size())
                .setOverwrite(overwrite);
        final Collection<GroupTemplate> groupsToAdd;

        // find existing groups with the same names and delete them if we're overwriting
        final BulkRemoveResult<String> bulkRemoveExistingResult = removeAllGroups(directoryId, groups, overwrite);
        if (overwrite) {
            groupsToAdd = Lists.newArrayList(getDeletedAndNotFoundEntities(groups, bulkRemoveExistingResult,
                    NameComparator.normaliserOf(GroupTemplate.class)));
        } else {
            groupsToAdd = Lists.newArrayList(getNotFoundEntities(groups, bulkRemoveExistingResult,
                    NameComparator.normaliserOf(GroupTemplate.class)));
        }

        // after the remove, any failures -> those groups are still there and can't be overwritten
        for (final String stillExistingGroupName : bulkRemoveExistingResult.getFailedEntities()) {
            try {
                bulkAddResultBuilder.addExistingEntity(findGroupByName(directoryId, stillExistingGroupName));
                if (overwrite) {
                    logger.error("Group could not be removed for bulk add with overwrite: {}", stillExistingGroupName);
                } else {
                    logger.info("Group already exists in directory; overwrite is off so import will skip it: {}",
                            stillExistingGroupName);
                }
            } catch (GroupNotFoundException e) {
                logger.debug(
                        "Group previously existed and couldn't be removed but is now missing so importing it: {}",
                        stillExistingGroupName);
                groupsToAdd.add(Iterables.find(groups, DirectoryEntityUtils.whereNameEquals(stillExistingGroupName)));
            }
        }

        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);

        // retain unique entities
        Set<GroupTemplate> uniqueGroupsToAdd = retainUniqueEntities(groupsToAdd);

        // perform the add operation
        final Collection<Group> successfulEntities;
        final Collection<Group> failedEntities;
        if (remoteDirectory instanceof InternalRemoteDirectory) {
            final BatchResult<Group> batchResult = ((InternalRemoteDirectory) remoteDirectory).addAllGroups(uniqueGroupsToAdd);
            successfulEntities = batchResult.getSuccessfulEntities();
            failedEntities = batchResult.getFailedEntities();
        } else {
            successfulEntities = new ArrayList<Group>(uniqueGroupsToAdd.size());
            failedEntities = new ArrayList<Group>();
            for (GroupTemplate group : uniqueGroupsToAdd) {
                try {
                    if (IdentifierUtils.hasLeadingOrTrailingWhitespace(group.getName())) {
                        throw new InvalidGroupException(group, "Group name may not contain leading or trailing whitespace");
                    }

                    successfulEntities.add(remoteDirectory.addGroup(group));
                } catch (Exception e) {
                    logger.error("Failed to add group: {}", group, e);
                    failedEntities.add(group);
                }
            }
        }

        bulkAddResultBuilder.addFailedEntities(failedEntities);
        logFailedEntities(remoteDirectory, failedEntities);

        // Fire events
        for (Group addedGroup : successfulEntities) {
            eventPublisher.publish(new GroupCreatedEvent(this, directory, addedGroup));
        }

        return bulkAddResultBuilder.build();
    }

    /**
     * Find and optionally (if {@code doRemove} is {@code true}) remove all of the given groups from the given directory.
     *
     * @param directoryId id of directory to remove groups from
     * @param groups      groups to find and possibly remove
     * @param doRemove    true to remove groups, false to just find them
     * @return a BulkRemoveResult with missingEntities containing those groups which were not found, and
     * failedEntities containing either the remainder (when doRemove is false) or those groups which were attempted to
     * be removed but could not be (when doRemove is true). In all cases, groups are recorded as their names which
     * will have been normalised according to the {@link NameComparator#normaliserOf(Class)} for {@link Group}.
     * @throws DirectoryPermissionException if directory does not have {@link OperationType#DELETE_GROUP} permission
     * @throws OperationFailedException     if the bulk delete failed
     * @throws DirectoryNotFoundException   if the directory with the given id could not be found
     */
    private BulkRemoveResult<String> removeAllGroups(final long directoryId, final Collection<GroupTemplate> groups, boolean doRemove)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        if (!permissionManager.hasPermission(directory, OperationType.DELETE_GROUP) && doRemove) {
            throw new DirectoryPermissionException("Directory does not allow removing groups");
        }

        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);

        List<GroupTemplate> groupsToRemove = new ArrayList<GroupTemplate>();
        BulkRemoveResult.Builder<String> resultBuilder = BulkRemoveResult.builder(groups.size());

        Function<Group, String> groupNameNormaliser = NameComparator.normaliserOf(Group.class);
        for (GroupTemplate group : groups) {
            try {
                findGroupByName(directoryId, groupNameNormaliser.apply(group));
                groupsToRemove.add(group);
            } catch (GroupNotFoundException e) {
                resultBuilder.addMissingEntity(groupNameNormaliser.apply(group));
            }
        }

        // retain unique entities
        Set<String> namesOfUniqueGroupsToRemove = retainUniqueEntities(
                Iterables.transform(groupsToRemove, groupNameNormaliser));

        if (!doRemove) {
            // all entities we would remove normally are instead "failed"
            resultBuilder.addFailedEntities(namesOfUniqueGroupsToRemove);
        } else {
            // perform the remove operation
            final Collection<String> successfulEntities;
            final Collection<String> failedEntities;
            if (remoteDirectory instanceof InternalRemoteDirectory) {
                BatchResult<String> batchResult = ((InternalRemoteDirectory) remoteDirectory).removeAllGroups(
                        namesOfUniqueGroupsToRemove);
                successfulEntities = batchResult.getSuccessfulEntities();
                failedEntities = batchResult.getFailedEntities();
            } else {
                successfulEntities = new ArrayList<String>(namesOfUniqueGroupsToRemove.size());
                failedEntities = new ArrayList<String>();
                for (String groupName : namesOfUniqueGroupsToRemove) {
                    try {
                        remoteDirectory.removeGroup(groupName);
                        successfulEntities.add(groupName);
                    } catch (Exception e) {
                        logger.error("Failed to remove group: {}", groupName, e);
                        failedEntities.add(groupName);
                    }
                }
            }

            resultBuilder.addFailedEntities(failedEntities);
            logFailedEntitiesWithNames(remoteDirectory, failedEntities);

            // Fire events
            for (String deletedGroup : successfulEntities) {
                eventPublisher.publish(new GroupDeletedEvent(this, directory, deletedGroup));
            }
        }

        return resultBuilder.build();
    }

    @Override
    public BulkAddResult<String> addAllUsersToGroup(final long directoryId, final Collection<String> userNames, final String groupName)
            throws DirectoryPermissionException, OperationFailedException, DirectoryNotFoundException, GroupNotFoundException {
        Directory directory = findDirectoryById(directoryId);

        // fail-fast if container not found (throws ONFE)
        final Group groupToUpdate = findGroupByName(directoryId, groupName);
        final OperationType operationType = getUpdateOperationType(groupToUpdate);

        if (permissionManager.hasPermission(directory, operationType)) {
            RemoteDirectory remoteDirectory = directoryInstanceLoader.getDirectory(directory);

            // build a list of memberships to add
            Set<String> usersToAdd = retainUniqueEntities(userNames);

            BulkAddResult.Builder<String> resultBuilder = BulkAddResult.<String>builder(userNames.size())
                    .setOverwrite(true);

            // bulk add
            if (remoteDirectory instanceof InternalRemoteDirectory) {
                final BatchResult<String> batchResult = ((InternalRemoteDirectory) remoteDirectory).addAllUsersToGroup(usersToAdd, groupName);
                final Collection<String> successfulUsers = batchResult.getSuccessfulEntities();
                for (final String username : batchResult.getFailedEntities()) {
                    if (isUserDirectGroupMember(directoryId, username, groupName)) {
                        resultBuilder.addExistingEntity(username);
                    } else {
                        resultBuilder.addFailedEntity(username);
                    }
                }

                // Fire events
                for (String username : successfulUsers) {
                    eventPublisher.publish(new GroupMembershipCreatedEvent(this, directory, username, groupName, MembershipType.GROUP_USER));
                }
                eventPublisher.publish(new GroupMembershipsCreatedEvent(this, directory, successfulUsers, groupName, MembershipType.GROUP_USER));

            } else {
                for (String username : usersToAdd) {
                    try {
                        // addUserToGroup will also publish events
                        addUserToGroup(directoryId, username, groupName);
                    } catch (MembershipAlreadyExistsException e) {
                        resultBuilder.addExistingEntity(username);
                    } catch (Exception e) {
                        resultBuilder.addFailedEntity(username);
                        logger.error(e.getMessage());
                    }
                }
            }

            final BulkAddResult<String> bulkAddResult = resultBuilder.build();

            for (String failedUser : bulkAddResult.getFailedEntities()) {
                logger.warn("Could not add the following user to the group [ {} ]: {}", groupName, failedUser);
            }

            return bulkAddResult;
        } else {
            if (operationType.equals(OperationType.UPDATE_GROUP)) {
                throw new DirectoryPermissionException("Directory does not allow group modifications");
            } else {
                throw new DirectoryPermissionException("Directory does not allow role modifications");
            }
        }
    }

    /**
     * Returns either CREATE_GROUP or CREATE_ROLE depending on the GroupType of the given Group
     *
     * @param group The Group
     * @return either CREATE_GROUP or CREATE_ROLE depending on the GroupType of the given Group
     */
    private static OperationType getCreateOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.CREATE_GROUP;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Returns either UPDATE_GROUP or UPDATE_ROLE depending on the GroupType of the given Group
     *
     * @param group The Group
     * @return either UPDATE_GROUP or UPDATE_ROLE depending on the GroupType of the given Group
     */
    private static OperationType getUpdateOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.UPDATE_GROUP;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Returns either UPDATE_GROUP_ATTRIBUTE or UPDATE_ROLE_ATTRIBUTE depending on the GroupType
     * of the given Group.
     *
     * @param group The Group
     * @return either UPDATE_GROUP_ATTRIBUTE or UPDATE_ROLE_ATTRIBUTE depending on the GroupType
     * of the given Group
     */
    private static OperationType getUpdateAttributeOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.UPDATE_GROUP_ATTRIBUTE;
            default:
                throw new UnsupportedOperationException();
        }
    }

    /**
     * Returns either DELETE_GROUP or DELETE_ROLE depending on the GroupType of the given Group
     *
     * @param group The Group
     * @return either DELETE_GROUP or DELETE_ROLE depending on the GroupType of the given Group
     */
    private static OperationType getDeleteOperationType(final Group group) {
        switch (group.getType()) {
            case GROUP:
                return OperationType.DELETE_GROUP;
            default:
                throw new UnsupportedOperationException();
        }
    }

    private static void logFailedEntities(RemoteDirectory remoteDirectory, final Collection<? extends DirectoryEntity> failedEntities) {
        logFailedEntitiesWithNames(remoteDirectory,
                Iterables.transform(failedEntities, new Function<DirectoryEntity, String>() {
                    @Override
                    public String apply(DirectoryEntity input) {
                        return input.getName();
                    }
                }));
    }

    private static void logFailedEntitiesWithNames(RemoteDirectory remoteDirectory, final Iterable<String> failedEntities) {
        String directoryName = remoteDirectory.getDescriptiveName();
        for (String failedEntity : failedEntities) {
            logger.warn("Could not add the following entityName to the directory [ {} ]: {}", directoryName, failedEntity);
        }
    }

    @Override
    public User findUserByExternalId(long directoryId, String externalId) throws DirectoryNotFoundException,
            UserNotFoundException, OperationFailedException {
        return getDirectoryImplementation(directoryId).findUserByExternalId(externalId);
    }

    @Override
    public UserWithAttributes findUserWithAttributesByExternalId(long directoryId, String externalId)
            throws DirectoryNotFoundException, UserNotFoundException, OperationFailedException {
        RemoteDirectory directory = getDirectoryImplementation(directoryId);
        User user = directory.findUserByExternalId(externalId); // if we ever add a findUserWithAttributesByExternalId, use it here
        return directory.findUserWithAttributesByName(user.getName());
    }

    @Override
    public void expireAllPasswords(long directoryId) throws OperationFailedException, DirectoryNotFoundException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);

        if (remoteDirectory.supportsPasswordExpiration()) {
            remoteDirectory.expireAllPasswords();
            eventPublisher.publish(new AllPasswordsExpiredEvent(this, findDirectoryById(directoryId)));
        } else {
            throw new OperationFailedException("Expiring passwords not supported by directory " + directoryId);
        }
    }

    @Override
    public boolean supportsExpireAllPasswords(long directoryId) throws DirectoryInstantiationException, DirectoryNotFoundException {
        RemoteDirectory remoteDirectory = getDirectoryImplementation(directoryId);
        return remoteDirectory.supportsPasswordExpiration();
    }

    @Override
    public AvatarReference getUserAvatarByName(long directoryId, String username, int sizeHint)
            throws UserNotFoundException, OperationFailedException, DirectoryNotFoundException {
        return getDirectoryImplementation(directoryId).getUserAvatarByName(username, sizeHint);
    }

    @Nonnull
    @Override
    public User findRemoteUserByName(Long directoryId, String username) throws OperationFailedException, DirectoryNotFoundException, UserNotFoundException {
        return getDirectoryImplementation(directoryId)
                .getAuthoritativeDirectory()
                .findUserByName(username);
    }

    @Override
    public User updateUserFromRemoteDirectory(User remoteUser) throws OperationFailedException, DirectoryNotFoundException, UserNotFoundException {
        return getDirectoryImplementation(remoteUser.getDirectoryId()).updateUserFromRemoteDirectory(remoteUser);
    }

    @Override
    public List<Application> findAuthorisedApplications(long directoryId, List<String> groupNames) {
        return applicationDAO.findAuthorisedApplications(directoryId, groupNames);
    }
}
