package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;

import java.util.List;

/**
 * A factory which may be able to create search strategies for a given list of active {@link Directory directories}
 *
 * @see MembershipSearchStrategy
 * @see UserSearchStrategy
 * @see GroupSearchStrategy
 */
public interface SearchStrategyFactory {
    /**
     * Attempts to create a {@link MembershipSearchStrategy} for the specified {@code isMembershipAggregationEnabled} and
     * associated {@code directories}.
     * <p>
     * If the factory cannot create a {@link MembershipSearchStrategy} for the specified membership aggregation
     * or directory configuration
     * {@code Optional.empty()} will be returned
     *
     * @param aggregateMemberships whether membership aggregation is enabled
     * @param directories the collection of directories to search
     * @return a {@link MembershipSearchStrategy} for the application and directories or {@code Optional.empty()}
     */
    MembershipSearchStrategy createMembershipSearchStrategy(
            boolean aggregateMemberships, List<Directory> directories, CanonicalityChecker canonicalityChecker, AccessFilter accessFilter);

    /**
     * Attempts to create a {@link UserSearchStrategy} for the specified {@code mergeUsers} and
     * associated {@code directories}.
     * <p>
     * If the factory cannot create a {@link UserSearchStrategy} for the specified membership aggregation
     * or directory configuration
     * {@code Optional.empty()} will be returned
     *
     * @param mergeUsers Whether users with the same name from different directories should be merged. Unmerged users
     *                   with the same name will be returned in the order defined by {@code directories} order.
     * @param directories the collection of directories to search
     * @return a {@link UserSearchStrategy} for the application and directories or {@code Optional.empty()}
     */
    UserSearchStrategy createUserSearchStrategy(boolean mergeUsers, List<Directory> directories, AccessFilter accessFilter);

    /**
     * Attempts to create a {@link GroupSearchStrategy} for the specified {@code mergeGroups} and
     * associated {@code directories}.
     * <p>
     * If the factory cannot create a {@link GroupSearchStrategy} for the specified membership aggregation
     * or directory configuration
     * {@code Optional.empty()} will be returned
     *
     * @param mergeGroups Whether groups with the same name should be merged. Unmerged groups with the same name
     *                    will be returned in the order defined by {@code directories} order.
     * @param directories the collection of directories to search
     * @return a {@link GroupSearchStrategy} for the application and directories or {@code Optional.empty()}
     */
    GroupSearchStrategy createGroupSearchStrategy(boolean mergeGroups, List<Directory> directories, AccessFilter accessFilter);
}
