package com.atlassian.crowd.manager.application.canonicality;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.search.EntityDescriptor;
import com.google.common.collect.Multimap;
import com.google.common.collect.SetMultimap;

import java.util.List;
import java.util.Set;

/**
 */
public interface CanonicalityChecker {
    /**
     * <p>
     * Given a username is duplicated in several {@link Directory user directories} under the same
     * {@link Application application}.
     * </p>
     * <p>The user in the first directory, according to directory ordering is considered to
     * be the canonical user for the given username, and the other users are shadowed and thus not to be returned from
     * searches.</p>
     *
     * This method removes all entities which were found for a given directory, but were not canonical in it (they existed
     * in a preceding directory, not necessarily in the allEntities map). The allEntities map will be mutated in the
     * process. For example:
     *
     * Given the following directories and entities:
     *
     * <ul>
     * <li>Directory 1 - Users A, B, C</li>
     * <li>Directory 2 - Users C, D, E</li>
     * </ul>
     *
     * At this point users A, B and C are known as canonical in the Directory 1 and will be removed from the
     * succeeding directories. This will leave users D and E in Directory 2. However their canonicality in Directory 2
     * is uncertain as they may exist in Directory 1, but may have not been matched by the query that was used to
     * retrieve the users. Because of this Directory 1 will be queried for users D and E and the results that were
     * found will be removed from all succeeding directories. If user D is found in Directory 1 this will give us the
     * following result:
     *
     * <ul>
     * <li>Directory 1 - Users A, B, C</li>
     * <li>Directory 2 - User E</li>
     * </ul>
     *
     * User D was not included in the result set due to being canonical in Directory 1, but not matching the query
     * used to retrieve the entities.
     *
     * @param allNames   a directoryId to list of entities map containing the entities to check for canonicality.
     * @param entity     the EntityDescriptor used to look up the entities that will be determined as canonical
     */

    void removeNonCanonicalEntities(final Multimap<Long, String> allNames, EntityDescriptor entity);

    /**
     * Returns names grouped by their canonical directory id.
     * Case-sensitivity of the provided names is preserved.
     * Names not found won't be returned.
     * @param names names to be grouped
     * @param entityDescriptor entity to which the names are referring to
     * @return names grouped by their canonical directory id
     */
    SetMultimap<Long, String> groupByCanonicalId(Set<String> names, EntityDescriptor entityDescriptor);

    /**
     * @return ordered list of directories for which the canonicality is computed
     */
    List<Directory> getDirectories();
}
