package com.atlassian.crowd.directory.loader;

import com.atlassian.crowd.directory.CachingDirectory;
import com.atlassian.crowd.directory.DbCachingRemoteDirectory;
import com.atlassian.crowd.directory.InternalRemoteDirectory;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.ldap.cache.CacheRefresherFactory;
import com.atlassian.crowd.directory.ldap.cache.DirectoryCacheFactory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.manager.audit.AuditService;
import com.atlassian.crowd.manager.audit.mapper.AuditLogGroupMapper;
import com.atlassian.crowd.manager.audit.mapper.AuditLogUserMapper;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.password.factory.PasswordEncoderFactory;
import com.atlassian.crowd.util.persistence.hibernate.batch.BatchConfigParser;
import com.atlassian.event.api.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;

/**
 * Loader that allows for proxying of a remote directory through a local cache/mirror. To enable proxying for a remote
 * directory you should not expose its loader directly to the Crowd service, but instead delegate through this loader.
 */
public class DbCachingRemoteDirectoryInstanceLoader extends AbstractDirectoryInstanceLoader implements InternalHybridDirectoryInstanceLoader {
    private static final Logger log = LoggerFactory.getLogger(DbCachingRemoteDirectoryInstanceLoader.class);

    private final InternalDirectoryInstanceLoader internalDirectoryInstanceLoader;
    private final DirectoryInstanceLoader remoteDirectoryInstanceLoader;
    private final DirectoryCacheFactory directoryCacheFactory;
    private final CacheRefresherFactory cacheRefresherFactory;
    protected final AuditService auditService;
    protected final AuditLogUserMapper auditLogUserMapper;
    protected final AuditLogGroupMapper auditLogGroupMapper;
    private final EventPublisher eventPublisher;
    private final DirectoryDao directoryDao;

    /**
     * Spring-friendly constructor.
     *
     * @param remoteDirectoryInstanceLoader   the remote directory instance loader.
     * @param internalDirectoryInstanceLoader the internal directory in which to do the caching
     * @param cacheRefresherFactory           cache refresher factory
     */
    public DbCachingRemoteDirectoryInstanceLoader(DirectoryInstanceLoader remoteDirectoryInstanceLoader,
                                                  InternalDirectoryInstanceLoader internalDirectoryInstanceLoader,
                                                  DirectoryCacheFactory directoryCacheFactory,
                                                  CacheRefresherFactory cacheRefresherFactory,
                                                  AuditService auditService, AuditLogUserMapper auditLogUserMapper,
                                                  AuditLogGroupMapper auditLogGroupMapper,
                                                  EventPublisher eventPublisher,
                                                  DirectoryDao directoryDao) {
        this.remoteDirectoryInstanceLoader = remoteDirectoryInstanceLoader;
        this.internalDirectoryInstanceLoader = internalDirectoryInstanceLoader;
        this.directoryCacheFactory = directoryCacheFactory;
        this.cacheRefresherFactory = cacheRefresherFactory;
        this.auditService = auditService;
        this.auditLogUserMapper = auditLogUserMapper;
        this.auditLogGroupMapper = auditLogGroupMapper;
        this.eventPublisher = eventPublisher;
        this.directoryDao = directoryDao;
    }

    @Override
    public RemoteDirectory getDirectory(final Directory directory) throws DirectoryInstantiationException {
        // make sure we get a brand new instance of the underlying directories (otherwise there may be a race condition on who gets the DirectoryUpdatedEvent first)
        final RemoteDirectory remoteDirectory = getRawDirectory(directory.getId(), directory.getImplementationClass(), directory.getAttributes());
        final InternalRemoteDirectory internalDirectory = getRawInternalDirectory(directory);

        return new DbCachingRemoteDirectory(remoteDirectory, internalDirectory,
                directoryCacheFactory, cacheRefresherFactory, auditService, auditLogUserMapper, directory.getName(),
                eventPublisher, directoryDao, new BatchConfigParser());
    }

    private InternalRemoteDirectory getRawInternalDirectory(Directory directory) throws DirectoryInstantiationException {
        DirectoryImpl internal = new DirectoryImpl(directory);

        // internal directory needs a password encoder (even if it's just to store blank passwords)
        final Map<String, String> newAttributes = new HashMap<String, String>(internal.getAttributes());
        newAttributes.put("user_encryption_method", PasswordEncoderFactory.ATLASSIAN_SECURITY_ENCODER);
        internal.setAttributes(newAttributes);

        return internalDirectoryInstanceLoader.getRawDirectory(directory.getId(), CachingDirectory.class.getName(), newAttributes);
    }

    /**
     * This method will NOT wire up the internal backed directory.
     * <p>
     * So no local groups or custom attributes.
     *
     * @param id         Directory ID
     * @param className  class name of directory.
     * @param attributes the configuration attributes to pass to the RemoteDirectory
     * @return directory without monitoring/caching and without backing internal directory.
     */
    public RemoteDirectory getRawDirectory(Long id, String className, Map<String, String> attributes) throws DirectoryInstantiationException {
        return remoteDirectoryInstanceLoader.getRawDirectory(id, className, attributes);
    }

    public boolean canLoad(String className) {
        return remoteDirectoryInstanceLoader.canLoad(className);
    }
}
