package com.atlassian.crowd.core.event.listener;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.directory.DirectoryPermissionException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class DirectoryDefaultGroupMembershipResolver implements DefaultGroupMembershipResolver {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryDefaultGroupMembershipResolver.class);
    public static final String AUTO_GROUPS_ADDED = "autoGroupsAdded";
    private final DirectoryManager directoryManager;

    public DirectoryDefaultGroupMembershipResolver(DirectoryManager directoryManager) {
        this.directoryManager = directoryManager;
    }

    @Override
    public Collection<String> getDefaultGroupNames(Application application, Directory directory, UserWithAttributes user) {
        if (Boolean.parseBoolean(user.getValue(AUTO_GROUPS_ADDED))) {
            return Collections.emptySet();
        }
        final List<String> groups = getDirectoryGroups(directory);
        if (!groups.isEmpty()) {
            logger.info("User '{}' will be added to default groups of directory '{}': {}", user.getName(), directory.getName(), groups);
        }
        return groups;
    }

    private List<String> getDirectoryGroups(Directory directory) {
        final String attributeValue = directory.getValue(DirectoryImpl.ATTRIBUTE_KEY_AUTO_ADD_GROUPS);
        if (attributeValue != null) {
            return Splitter.on(DirectoryImpl.AUTO_ADD_GROUPS_SEPARATOR).splitToList(attributeValue);
        }
        return ImmutableList.of();
    }

    @Override
    public void onDefaultGroupsAdded(Application application, Directory directory, UserWithAttributes user) throws OperationFailedException {
        try {
            directoryManager.storeUserAttributes(directory.getId(), user.getName(), ImmutableMap.of(AUTO_GROUPS_ADDED, Collections.singleton(Boolean.TRUE.toString())));
        } catch (DirectoryPermissionException | DirectoryNotFoundException | UserNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }
}