package com.atlassian.crowd.manager.directory.monitor;

import com.atlassian.crowd.event.application.ApplicationReadyEvent;
import com.atlassian.crowd.event.directory.DirectoryCreatedEvent;
import com.atlassian.crowd.event.directory.DirectoryDeletedEvent;
import com.atlassian.crowd.event.directory.DirectoryUpdatedEvent;
import com.atlassian.crowd.event.migration.XMLRestoreFinishedEvent;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.RunMode;
import com.atlassian.scheduler.config.Schedule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Handles scheduling {@link DirectoryMonitorRefresherJob} when the application is first started, restored, or when the
 * directory configuration changes
 */
public final class DirectoryMonitorRefresherStarter {
    static final JobId JOB_ID = JobId.of(DirectoryMonitorRefresherStarter.class.getName() + "-job");

    private static final Logger log = LoggerFactory.getLogger(DirectoryMonitorRefresherStarter.class);
    private final EventPublisher eventPublisher;
    private final SchedulerService schedulerService;
    private final long refresherJobIntervalMillis;

    public DirectoryMonitorRefresherStarter(final EventPublisher eventPublisher,
                                            final SchedulerService schedulerService,
                                            final long refresherJobIntervalMillis) {
        this.eventPublisher = checkNotNull(eventPublisher);
        this.schedulerService = schedulerService;
        this.refresherJobIntervalMillis = refresherJobIntervalMillis;
    }

    @PostConstruct
    public void registerListener() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void unregisterListener() {
        eventPublisher.unregister(this);
    }

    @EventListener
    public void onApplicationReady(ApplicationReadyEvent event) {
        triggerDirectoryMonitoringJob(event);
    }

    @EventListener
    public void onXMLRestore(XMLRestoreFinishedEvent event) {
        triggerDirectoryMonitoringJob(event);
    }

    @EventListener
    public void handleEvent(DirectoryUpdatedEvent event) {
        triggerDirectoryMonitoringJob(event);
    }

    @EventListener
    public void handleEvent(DirectoryDeletedEvent event) {
        triggerDirectoryMonitoringJob(event);
    }

    @EventListener
    public void handleEvent(DirectoryCreatedEvent event) {
        triggerDirectoryMonitoringJob(event);
    }


    private void triggerDirectoryMonitoringJob(Object cause) {
        try {
            log.debug("Rescheduling directory monitoring job due to {}", cause);
            schedulerService.scheduleJob(JOB_ID,
                    JobConfig.forJobRunnerKey(DirectoryMonitorRefresherJob.JOB_RUNNER_KEY)
                            .withRunMode(RunMode.RUN_ONCE_PER_CLUSTER)
                            .withSchedule(Schedule.forInterval(refresherJobIntervalMillis, null))
            );
        } catch (SchedulerServiceException e) {
            log.warn("Failed to reschedule directory monitoring job", e);
        }
    }
}
