package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * Wraps {@link DirectoryDao} calls in a {@link Transactional}. This is helpful when another component needs to
 * both define a {@link DirectoryDao} method in as transactional and call it.
 */
@Transactional
public class DirectoryDaoTransactionalDecorator implements TransactionalDirectoryDao {

    private final DirectoryDao dao;

    public DirectoryDaoTransactionalDecorator(DirectoryDao dao) {
        this.dao = dao;
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#findById(long)}
     *
     * @param directoryId dir id
     * @return directory
     * @throws DirectoryNotFoundException if directory is not present
     */
    @Override
    @Transactional(readOnly = true)
    public Directory findById(final long directoryId) throws DirectoryNotFoundException {
        return dao.findById(directoryId);
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#update(Directory)}
     *
     * @param directory directory to be updated
     * @throws DirectoryNotFoundException if directory is not present
     */
    @Override
    public Directory update(Directory directory) throws DirectoryNotFoundException {
        return dao.update(directory);
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#findByName(String)}
     *
     * @param name directory name
     * @throws DirectoryNotFoundException if directory is not present
     */
    @Override
    @Transactional(readOnly = true)
    public Directory findByName(String name) throws DirectoryNotFoundException {
        return dao.findByName(name);
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#add(Directory)}
     * @param directory the directory to persist
     * @return the newly-persisted directory, which should be used for subsequent operations
     */
    @Override
    public Directory add(Directory directory) {
        return dao.add(directory);
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#remove(Directory)}
     * @param directory the directory to remove
     * @throws DirectoryNotFoundException if the directory does not exist
     */
    @Override
    public void remove(Directory directory) throws DirectoryNotFoundException {
        dao.remove(directory);
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#findAll()}
     * @return all directories
     * @deprecated Use {@link #search(EntityQuery)} instead. Since v3.2.0
     */
    @Override
    @Deprecated
    @Transactional(readOnly = true)
    public List<Directory> findAll() {
        return dao.findAll();
    }

    /**
     * Makes a transactional call to {@link DirectoryDao#search(EntityQuery)}
     * @param entityQuery the search query to run against the directory data store
     * @return a list of directories matching the query
     * @see com.atlassian.crowd.search.builder.QueryBuilder#queryFor
     */
    @Override
    @Transactional(readOnly = true)
    public List<Directory> search(EntityQuery<Directory> entityQuery) {
        return dao.search(entityQuery);
    }
}
