package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.manager.application.canonicality.SimpleCanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.query.QueryUtils;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.google.common.base.Preconditions;

import java.util.List;

/**
 * An {@link UserSearchStrategy} and {@link GroupSearchStrategy} implementation which merges results in-memory.
 * <p>
 * This class will search across multiple directories, potentially consuming a lot of memory.
 * This is the same (in spirit) as what Crowd 2.8 and earlier would do by default.
 */
public class InMemoryEntitySearchStrategy implements UserSearchStrategy, GroupSearchStrategy {
    protected final DirectoryManagerSearchWrapper directoryManagerSearchWrapper;
    protected final List<Directory> directories;
    private final boolean mergeEntities;
    private final AccessFilter accessFilter;
    private final CanonicalityChecker canonicalityChecker;

    public InMemoryEntitySearchStrategy(DirectoryManager directoryManager, List<Directory> directories, boolean mergeEntities, AccessFilter accessFilter) {
        this.directoryManagerSearchWrapper = new DirectoryManagerSearchWrapper(directoryManager);
        this.directories = directories;
        this.mergeEntities = mergeEntities;
        this.accessFilter = accessFilter;
        this.canonicalityChecker = new SimpleCanonicalityChecker(directoryManager, directories);
    }

    @Override
    public <T> List<T> searchUsers(final EntityQuery<T> query) {
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, User.class);
        Preconditions.checkArgument(query.getEntityDescriptor().getEntityType() == Entity.USER);

        return search(query);
    }

    @Override
    public <T> List<T> searchGroups(final EntityQuery<T> query) {
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class);
        Preconditions.checkArgument(query.getEntityDescriptor().getEntityType() == Entity.GROUP);

        return search(query);
    }

    protected <T> List<T> search(final EntityQuery<T> query) {
        return InMemoryQueryRunner.createEntityQueryRunner(
                directoryManagerSearchWrapper, directories,
                requiresCanonicalityFiltering(query) ? canonicalityChecker : null,
                mergeEntities,
                accessFilter::getDirectoryQueryWithFilter,
                query)
                .search();
    }

    private boolean requiresCanonicalityFiltering(final EntityQuery<?> query) {
        // We don't check groups canonicality. Group that is shadowed can still be an effective parent of
        // canonical groups.
        return mergeEntities
                && query.getEntityDescriptor().getEntityType() == Entity.USER
                && accessFilter.requiresFiltering(Entity.USER);
    }
}
