package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.directory.DirectoryManager;

import java.util.List;

import static java.util.Objects.requireNonNull;

/**
 * A {@link SearchStrategyFactory} which will delegate to the {@link DirectoryManager} for all of its
 * {@link MembershipSearchStrategy search strategies}.
 * <p>
 * This factory is guaranteed to return non-empty for any application
 * and configuration of directories
 *
 * @see NoDirectorySearchStrategy
 * @see InMemoryAggregatingMembershipSearchStrategy
 * @since 2.9
 */
public class DefaultSearchStrategyFactory implements SearchStrategyFactory {
    private final DirectoryManager directoryManager;

    public DefaultSearchStrategyFactory(DirectoryManager directoryManager) {
        this.directoryManager = requireNonNull(directoryManager, "directoryManager");
    }

    @Override
    public MembershipSearchStrategy createMembershipSearchStrategy(
            boolean aggregateMemberships,
            List<Directory> directories,
            CanonicalityChecker canonicalityChecker,
            AccessFilter filter) {
        if (directories.isEmpty()) {
            return NoDirectorySearchStrategy.INSTANCE;
        }
        return aggregateMemberships ? new InMemoryAggregatingMembershipSearchStrategy(directoryManager, directories, filter)
                : new InMemoryNonAggregatingMembershipSearchStrategy(directoryManager, directories, canonicalityChecker, filter);
    }

    @Override
    public UserSearchStrategy createUserSearchStrategy(boolean mergeUsers, List<Directory> directories, AccessFilter filter) {
        if (directories.isEmpty()) {
            return NoDirectorySearchStrategy.INSTANCE;
        }
        return new InMemoryEntitySearchStrategy(directoryManager, directories, mergeUsers, filter);
    }

    @Override
    public GroupSearchStrategy createGroupSearchStrategy(boolean mergeGroups, List<Directory> directories, AccessFilter filter) {
        if (directories.isEmpty()) {
            return NoDirectorySearchStrategy.INSTANCE;
        }
        return new InMemoryEntitySearchStrategy(directoryManager, directories, mergeGroups, filter);
    }
}
