package com.atlassian.crowd.manager.application.search;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierSet;
import com.atlassian.crowd.manager.application.canonicality.CanonicalityChecker;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Lists;

import java.util.List;
import java.util.function.BiFunction;

/**
 * An abstract {@link MembershipSearchStrategy} which searches across multiple directories in memory for users and groups
 * <p>
 * This is considered the worse case {@link MembershipSearchStrategy} to use as aggregation across multiple directories
 * will be done in-memory, potentially consuming a lot of memory. This is the same (in spirit) as what Crowd 2.8
 * and earlier would do by default.
 *
 * @since 2.9
 */
public abstract class AbstractInMemoryMembershipSearchStrategy implements MembershipSearchStrategy {
    protected final DirectoryManagerSearchWrapper directoryManagerSearchWrapper;
    protected final List<Directory> directories;
    protected final List<Long> directoryIds;
    protected final AccessFilter accessFilter;

    public AbstractInMemoryMembershipSearchStrategy(DirectoryManager directoryManager, List<Directory> directories, AccessFilter accessFilter) {
        this.directoryManagerSearchWrapper = new DirectoryManagerSearchWrapper(directoryManager);
        this.directories = ImmutableList.copyOf(directories);
        this.directoryIds = ImmutableList.copyOf(Lists.transform(directories, Directory::getId));
        this.accessFilter = accessFilter;
    }

    @Override
    public <T> List<T> searchDirectGroupRelationships(MembershipQuery<T> query) {
        return searchGroupRelationships(query, false);
    }

    @Override
    public <T> List<T> searchNestedGroupRelationships(MembershipQuery<T> query) {
        return searchGroupRelationships(query, true);
    }

    protected <T> List<T> searchGroupRelationships(MembershipQuery<T> query, boolean nested) {
        return createSearcher(query, nested).search();
    }

    @Override
    public <T> ListMultimap<String, T> searchDirectGroupRelationshipsGroupedByName(MembershipQuery<T> query) {
        return createSearcher(query, false).searchGroupedByName();
    }

    private <T> InMemoryQueryRunner<T, MembershipQuery<T>> createSearcher(MembershipQuery<T> original, boolean nested) {
        BiFunction<Directory, MembershipQuery<T>, MembershipQuery<T>> transformer = getQueryTransformer(original);
        return InMemoryQueryRunner.createMembershipQueryRunner(
                directoryManagerSearchWrapper, directories,
                getCanonicalityCheckerIfNeeded(original),
                true,
                nested,
                (directory, query) -> accessFilter.getDirectoryQueryWithFilter(directory, transformer.apply(directory, query))
                        .map(q -> filterOriginalResultsIfNeeded(original, nested, q)),
                original);
    }

    // This is for backward compatibility. Nested members query does not return original entities
    // from original.getEntityNamesToMatch(), even if some of them are parents/children of the others.
    private <T> DirectoryQueryWithFilter<T> filterOriginalResultsIfNeeded(MembershipQuery<T> query,
                                                                          boolean nested,
                                                                          DirectoryQueryWithFilter<T> directoryQueryWithFilter) {
        if (nested
                && query.getEntityToReturn().getEntityType() == Entity.GROUP
                && query.getEntityToMatch().getEntityType() == Entity.GROUP) {
            MembershipQuery<T> transformed = directoryQueryWithFilter.getMembershipQuery();
            IdentifierSet diff = IdentifierSet.difference(query.getEntityNamesToMatch(), transformed.getEntityNamesToMatch());
            if (!diff.isEmpty()) {
                return new DirectoryQueryWithFilter<>(
                        directoryQueryWithFilter.getDirectory(),
                        transformed.baseSplitQuery().addToMaxResults(diff.size()),
                        results -> NamesUtil.filterOutByName(directoryQueryWithFilter.filterResults(results), diff::contains));
            }
        }
        return directoryQueryWithFilter;
    }

    protected abstract CanonicalityChecker getCanonicalityCheckerIfNeeded(MembershipQuery<?> query);

    protected abstract <T> BiFunction<Directory, MembershipQuery<T>, MembershipQuery<T>> getQueryTransformer(MembershipQuery<T> original);
}
