package com.atlassian.crowd.manager.application.canonicality;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.impl.IdentifierSet;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.manager.application.search.DirectoryManagerSearchWrapper;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.NullRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Table;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;

/**
 * Simple implementation of {@link CanonicalityChecker}.
 * This implementation is not efficient for multiple calls, as it fetches required data incrementally.
 */
public class SimpleCanonicalityChecker implements CanonicalityChecker {
    protected static final int BATCH_SIZE = 1000;

    private final DirectoryManagerSearchWrapper directoryManagerSearchWrapper;
    private final List<Directory> directories;
    private final Table<EntityDescriptor, String, Long> nameAndEntityToCanonicalDirId = HashBasedTable.create();
    private final Table<EntityDescriptor, Long, IdentifierSet> allEntities = HashBasedTable.create();
    private final int batchSize;

    public SimpleCanonicalityChecker(DirectoryManager directoryManager, List<Directory> directories) {
        this(directoryManager, directories, BATCH_SIZE);
    }

    @VisibleForTesting
    SimpleCanonicalityChecker(DirectoryManager directoryManager, List<Directory> directories, int batchSize) {
        this.directoryManagerSearchWrapper = new DirectoryManagerSearchWrapper(directoryManager);
        this.directories = ImmutableList.copyOf(directories);
        this.batchSize = batchSize;
    }

    @Override
    public void removeNonCanonicalEntities(final Multimap<Long, String> allNames, final EntityDescriptor entityDescriptor) {
        final Map<String, Long> nameToCanonicalDirId = nameAndEntityToCanonicalDirId.row(entityDescriptor);
        computeMissing(allNames.values(), entityDescriptor, (directory, unknown) -> {
            final long dirId = directory.getId();
            for (final Iterator<String> it = allNames.get(dirId).iterator(); it.hasNext();) {
                final String lowerName = IdentifierUtils.toLowerCase(it.next());
                unknown.removeAll(lowerName);
                final long canonicalDirId = nameToCanonicalDirId.computeIfAbsent(lowerName, ignore -> dirId);
                if (canonicalDirId != dirId) {
                    it.remove();
                }
            }
        });
    }

    @Override
    public SetMultimap<Long, String> groupByCanonicalId(final Set<String> names, final EntityDescriptor entityDescriptor) {
        computeMissing(names, entityDescriptor, (dir, unknown) -> {});
        final Map<String, Long> nameToCanonicalDirId = this.nameAndEntityToCanonicalDirId.row(entityDescriptor);
        final SetMultimap result = HashMultimap.create();
        for (String name : names) {
            final Long dirId = nameToCanonicalDirId.get(IdentifierUtils.toLowerCase(name));
            if (dirId != null) {
                result.put(dirId, name);
            }
        }
        return result;
    }

    private void computeMissing(final Collection<String> names,
                                final EntityDescriptor entityDescriptor,
                                final BiConsumer<Directory, SetMultimap<String, String>> consumer) {
        final Map<String, Long> nameToCanonicalDirId = nameAndEntityToCanonicalDirId.row(entityDescriptor);
        final SetMultimap<String, String> unknown = HashMultimap.create(
                Multimaps.index(names, IdentifierUtils::toLowerCase));
        unknown.keySet().removeAll(nameToCanonicalDirId.keySet());
        for (Directory directory : directories) {
            consumer.accept(directory, unknown);
            final IdentifierSet found = findEntitiesInternal(unknown.values(), entityDescriptor, directory);
            unknown.keySet().removeAll(found);
            found.forEach(name -> nameToCanonicalDirId.put(name, directory.getId()));
        }
    }

    private IdentifierSet findEntitiesInternal(Collection<String> candidates, EntityDescriptor entity, Directory directory) {
        Preconditions.checkArgument(entity.equals(EntityDescriptor.user()) || entity.equals(EntityDescriptor.group()));
        if (candidates.isEmpty()) {
            return IdentifierSet.EMPTY;
        }
        final EntityQuery<String> allNamesQuery = QueryBuilder.queryFor(
                String.class, entity, NullRestriction.INSTANCE, 0, EntityQuery.ALL_RESULTS);
        IdentifierSet allEntities = this.allEntities.get(entity, directory.getId());
        if (allEntities == null && candidates.size() > batchSize) {
            allEntities = new IdentifierSet(directoryManagerSearchWrapper.search(directory.getId(), allNamesQuery));
            this.allEntities.put(entity, directory.getId(), allEntities);
        }
        if (allEntities != null) {
            return IdentifierSet.intersection(allEntities, candidates);
        }
        final boolean isUserQuery = entity.equals(EntityDescriptor.user());
        final SearchRestriction restriction = Restriction.on(isUserQuery ? UserTermKeys.USERNAME : GroupTermKeys.NAME)
                .exactlyMatchingAny(candidates);
        return new IdentifierSet(directoryManagerSearchWrapper.search(
                directory.getId(), allNamesQuery.withSearchRestriction(restriction)));
    }

    public List<Directory> getDirectories() {
        return directories;
    }
}
