package com.atlassian.crowd.manager.application.canonicality;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.application.filtering.AccessFilter;
import com.atlassian.crowd.manager.application.filtering.AccessFilters;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.Entity;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Optional;

/**
 * Helper class for getting canonical entity from multiple directories. "fastFail" methods re-throw
 * {@link OperationFailedException}, other methods ignore it.
 */
public class CanonicalEntityByNameFinder {
    private static final Logger logger = LoggerFactory.getLogger(CanonicalEntityByNameFinder.class);

    private static final String LOCATED_USER_IN_DIRECTORY_WITH_ACCESS_LOG = "Located user '{}' in directory {} '{}'";
    private static final String LOCATED_USER_IN_DIRECTORY_WITHOUT_ACCESS_LOG = LOCATED_USER_IN_DIRECTORY_WITH_ACCESS_LOG
            + ", but null returned as the user does not have access to the application.";

    private final DirectoryManager directoryManager;
    private final List<Directory> directories;
    private final AccessFilter accessFilter;

    public CanonicalEntityByNameFinder(DirectoryManager directoryManager, Iterable<Directory> directories) {
        this(directoryManager, directories, AccessFilters.UNFILTERED);
    }

    public CanonicalEntityByNameFinder(DirectoryManager directoryManager, Iterable<Directory> directories, AccessFilter accessFilter) {
        this.directoryManager = directoryManager;
        this.directories = ImmutableList.copyOf(directories);
        this.accessFilter = accessFilter;
    }

    public Group fastFailingFindGroupByName(String name) throws GroupNotFoundException, OperationFailedException {
        return fastFailingFindOptionalGroupByName(name)
                .orElseThrow(() -> new GroupNotFoundException(name));
    }

    public User fastFailingFindUserByName(String name) throws UserNotFoundException, OperationFailedException {
        return fastFailingFindOptionalUserByName(name)
                .orElseThrow(() -> new UserNotFoundException(name));
    }

    public Optional<Group> fastFailingFindOptionalGroupByName(String name) throws OperationFailedException {
        return fastFailFindByName(directoryManager::findGroupByName, name);
    }

    public Optional<User> fastFailingFindOptionalUserByName(String name) throws OperationFailedException {
        return fastFailFindByName(directoryManager::findUserByName, name);
    }

    public Group findGroupByName(String name) throws GroupNotFoundException {
        return findByName(directoryManager::findGroupByName, name)
                .orElseThrow(() -> new GroupNotFoundException(name));
    }

    public GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException {
        return findByName(directoryManager::findGroupWithAttributesByName, name)
                .orElseThrow(() -> new GroupNotFoundException(name));
    }

    public User findUserByName(String name) throws UserNotFoundException {
        return findByName(directoryManager::findUserByName, name)
                .orElseThrow(() -> new UserNotFoundException(name));
    }

    public User findRemoteUserByName(String name) throws UserNotFoundException {
        return findByName(directoryManager::findRemoteUserByName, name)
                .orElseThrow(() -> new UserNotFoundException(name));
    }

    public UserWithAttributes findUserWithAttributesByName(final String name) throws UserNotFoundException {
        return findByName(directoryManager::findUserWithAttributesByName, name)
                .orElseThrow(() -> new UserNotFoundException(name));
    }

    private <T extends DirectoryEntity> Optional<T> findByName(Searcher<T> searcher, String name) {
        try {
            return findByName(searcher, name, false);
        } catch (OperationFailedException e) {
            throw new com.atlassian.crowd.exception.runtime.OperationFailedException(e);
        }
    }

    private <T extends DirectoryEntity> Optional<T> fastFailFindByName(Searcher<T> searcher, String name) throws OperationFailedException {
        return findByName(searcher, name, true);
    }

    private <T extends DirectoryEntity> Optional<T> findByName(Searcher<T> searcher, String name, boolean failFast)
            throws OperationFailedException {
        for (final Directory directory : directories) {
            try {
                T entity = searcher.findByName(directory.getId(), name);
                if (entity instanceof User) {
                    Optional<T> userWithAccess = accessFilter.hasAccess(entity.getDirectoryId(), Entity.USER, entity.getName())
                            ? Optional.of(entity) : Optional.empty();
                    logger.trace(userWithAccess.isPresent()
                            ? LOCATED_USER_IN_DIRECTORY_WITH_ACCESS_LOG
                            : LOCATED_USER_IN_DIRECTORY_WITHOUT_ACCESS_LOG,
                            entity.getName(), directory.getId(), directory.getName());
                    return userWithAccess;
                } else if (accessFilter.hasAccess(entity.getDirectoryId(), Entity.GROUP, entity.getName())) {
                    return Optional.of(entity);
                }
                // keep cycling, shadowed group can still be an effective parent of canonical groups.
            } catch (final ObjectNotFoundException e) {
                // entity not in directory, keep cycling
            } catch (final OperationFailedException e) {
                if (failFast) {
                    throw e;
                } else {
                    // directory has some massive error, keep cycling
                    logger.error(e.getMessage(), e);
                }
            } catch (DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryIteration(e);
            }
        }
        return Optional.empty();
    }

    private interface Searcher<T extends DirectoryEntity> {
        T findByName(long dirId, String name) throws DirectoryNotFoundException, OperationFailedException, ObjectNotFoundException;
    }

    private static ConcurrentModificationException concurrentModificationExceptionForDirectoryIteration(
            DirectoryNotFoundException e) {
        return new ConcurrentModificationException("Directory mapping was removed while iterating through directories", e);
    }
}
