package com.atlassian.crowd.event;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimestampBasedEventToken {
    static final String VERSION_STAMP = "1";
    private static final Logger log = LoggerFactory.getLogger(TimestampBasedEventToken.class);
    final long timestamp;
    final List<Long> dirIds;

    public TimestampBasedEventToken(long timestamp, List<Long> dirIds) {
        this.timestamp = timestamp;
        this.dirIds = dirIds;
    }

    public static Optional<TimestampBasedEventToken> unmarshall(String marshalled) {
        final String[] tokens = marshalled.split("\\.");
        if (tokens.length != 3 || !Objects.equals(tokens[0], VERSION_STAMP)) {
            log.debug("Unknown token format for version {}: {}", VERSION_STAMP, marshalled);
            return Optional.empty();
        }

        try {
            final Long timestamp = Long.valueOf(tokens[1]);
            final List<Long> dirIds = Arrays.stream(tokens[2].split("-")).map(Long::valueOf).collect(Collectors.toList());
            return Optional.of(new TimestampBasedEventToken(timestamp, dirIds));
        } catch (NumberFormatException e) {
            log.debug("Error parsing token", e);
            return Optional.empty();
        }
    }

    public String marshall() {
        return String.format("%s.%s.%s", VERSION_STAMP,
                String.valueOf(timestamp),
                dirIds.stream().map(String::valueOf).collect(Collectors.joining("-")));
    }

    public long getTimestamp() {
        return timestamp;
    }

    public List<Long> getDirIds() {
        return dirIds;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("timestamp", timestamp)
                .add("dirIds", dirIds)
                .toString();
    }
}