package com.atlassian.crowd.directory;

import com.atlassian.crowd.directory.DirectoryCacheChangeOperations.AddRemoveSets;
import com.atlassian.crowd.directory.DirectoryCacheChangeOperations.GroupsToAddUpdateReplace;
import com.atlassian.crowd.directory.synchronisation.cache.DirectoryCache;
import com.atlassian.crowd.directory.synchronisation.utils.AddUpdateSets;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.util.TimedOperation;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public class DirectoryCacheImplUsingChangeOperations implements DirectoryCache {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryCacheImplUsingChangeOperations.class);

    private final DirectoryCacheChangeOperations dc;

    public DirectoryCacheImplUsingChangeOperations(DirectoryCacheChangeOperations dc) {
        this.dc = dc;
    }

    @Override
    public AddUpdateSets<UserTemplateWithCredentialAndAttributes, UserTemplate> addOrUpdateCachedUsers(final Collection<? extends User> remoteUsers, final Date syncStartDate)
            throws OperationFailedException {
        TimedOperation addOrUpdateRemoteUsersOperation = new TimedOperation();
        try {
            Set<UserTemplateWithCredentialAndAttributes> usersToAdd;
            Set<UserTemplate> usersToUpdate;

            TimedOperation findingUsersToAddAndUpdateOperation = new TimedOperation();
            AddUpdateSets<UserTemplateWithCredentialAndAttributes, UserTemplate> result;
            try {
                result = dc.getUsersToAddAndUpdate(remoteUsers, syncStartDate);
                usersToAdd = result.getToAddSet();
                usersToUpdate = result.getToUpdateSet();
            } finally {
                logger.info(findingUsersToAddAndUpdateOperation.complete("scanned and compared [ " + remoteUsers.size() + " ] users for update in DB cache"));
            }

            // updates (including renames) must happen before additions to avoid clashes with renamed-and-replaced users
            dc.updateUsers(usersToUpdate);
            dc.addUsers(usersToAdd);
            return result;
        } finally {
            logger.info(addOrUpdateRemoteUsersOperation.complete("synchronised [ " + remoteUsers.size() + " ] users"));
        }
    }

    @Override
    public void deleteCachedUsersByGuid(Set<String> guids) throws OperationFailedException {
        dc.deleteCachedUsersByGuid(guids);
    }

    @Override
    public void addOrUpdateCachedGroups(final Collection<? extends Group> remoteGroups, final Date syncStartDate) throws OperationFailedException {
        logger.info("scanning [ {} ] groups to add or update", remoteGroups.size());

        TimedOperation operation = new TimedOperation();
        try {
            GroupsToAddUpdateReplace addUpdateReplace = dc.findGroupsToUpdate(remoteGroups, syncStartDate);

            logger.debug("replacing [ {} ] groups", addUpdateReplace.groupsToReplace.size());
            dc.removeGroups(addUpdateReplace.groupsToReplace.keySet());

            Set<GroupTemplate> allToAdd = new HashSet<>();
            allToAdd.addAll(addUpdateReplace.groupsToAdd);
            allToAdd.addAll(addUpdateReplace.groupsToReplace.values());
            dc.addGroups(allToAdd);

            dc.updateGroups(addUpdateReplace.groupsToUpdate);
        } finally {
            logger.info(operation.complete("synchronized [ " + remoteGroups.size() + " ] groups"));
        }
    }

    @Override
    public void deleteCachedGroupsNotIn(final GroupType groupType, final List<? extends Group> remoteGroups, final Date syncStartDate) throws OperationFailedException {
        dc.deleteCachedGroupsNotIn(groupType, remoteGroups, syncStartDate);
    }

    @Override
    public void deleteCachedGroupsNotInByExternalId(final Collection<? extends Group> remoteGroups, Date syncStartDate) throws OperationFailedException {
        dc.deleteCachedGroupsNotInByExternalId(remoteGroups, syncStartDate);
    }

    @Override
    public void syncUserMembersForGroup(final Group parentGroup, final Collection<String> remoteUsers) throws OperationFailedException {
        if (shouldSkipSyncingGroupMembers(parentGroup)) {
            return;
        }

        TimedOperation operation = new TimedOperation();
        try {
            AddRemoveSets<String> addRemove = dc.findUserMembershipForGroupChanges(parentGroup, remoteUsers);

            logger.debug("removing [ {} ] users from group [ {} ]", addRemove.toRemove.size(), parentGroup.getName());
            dc.removeUserMembershipsForGroup(parentGroup, addRemove.toRemove);
            logger.debug("adding [ {} ] users to group [ {} ]", addRemove.toAdd.size(), parentGroup.getName());
            dc.addUserMembershipsForGroup(parentGroup, addRemove.toAdd);
        } finally {
            logger.debug(operation.complete("synchronised [ " + remoteUsers.size() + " ] user members for group [ " + parentGroup.getName() + " ]"));
        }
    }

    @Override
    public void addUserMembersForGroup(final Group parentGroup, final Set<String> remoteUsers) throws OperationFailedException {
        if (shouldSkipSyncingGroupMembers(parentGroup)) {
            return;
        }
        TimedOperation operation = new TimedOperation();
        try {
            dc.addUserMembershipsForGroup(parentGroup, remoteUsers);
        } finally {
            logger.debug(operation.complete("added [ " + remoteUsers.size() + " ] user members for group [ " + parentGroup.getName() + " ]"));
        }
    }

    @Override
    public void deleteUserMembersForGroup(final Group parentGroup, final Set<String> remoteUsers) throws OperationFailedException {
        if (shouldSkipSyncingGroupMembers(parentGroup)) {
            return;
        }

        TimedOperation operation = new TimedOperation();
        try {
            dc.removeUserMembershipsForGroup(parentGroup, remoteUsers);
        } finally {
            logger.debug(operation.complete("removed [ " + remoteUsers.size() + " ] user members for group [ " + parentGroup.getName() + " ]"));
        }
    }

    @Override
    public void syncGroupMembersForGroup(final Group parentGroup, final Collection<String> remoteGroups) throws OperationFailedException {
        if (shouldSkipSyncingGroupMembers(parentGroup)) {
            return;
        }

        TimedOperation operation = new TimedOperation();
        try {
            AddRemoveSets<String> addRemove = dc.findGroupMembershipForGroupChanges(parentGroup, remoteGroups);


            logger.debug("removing [ " + addRemove.toRemove.size() + " ] group members to group [ " + parentGroup.getName() + " ]");
            dc.removeGroupMembershipsForGroup(parentGroup, addRemove.toRemove);
            logger.debug("adding [ " + addRemove.toAdd.size() + " ] group members from group [ " + parentGroup.getName() + " ]");
            dc.addGroupMembershipsForGroup(parentGroup, addRemove.toAdd);
        } finally {
            logger.debug(operation.complete("synchronised [ " + remoteGroups.size() + " ] group members for group [ " + parentGroup.getName() + " ]"));
        }
    }

    @VisibleForTesting
    boolean shouldSkipSyncingGroupMembers(final Group parentGroup) throws OperationFailedException {
        final DirectoryCacheChangeOperations.GroupShadowingType groupShadowingType = dc.isGroupShadowed(parentGroup);
        switch(groupShadowingType) {
            case SHADOWED_BY_LOCAL_GROUP:
                logger.info("Skipping update of group {} due to the group being shadowed by a local group with the same name", parentGroup.getName());
                return true;
            case SHADOWED_BY_ROLE:
                logger.info("Skipping update of group {} due to the group being shadowed by a {} with the same name", parentGroup.getName(), GroupType.LEGACY_ROLE);
                return true;
            case GROUP_REMOVED:
                logger.info("Skipping update of group {} due to the group being removed from the cache in the mean time.", parentGroup.getName());
                return true;
            default:
                return false;
        }
    }

    @Override
    public void addGroupMembersForGroup(final Group parentGroup, final Set<String> remoteGroups) throws OperationFailedException {
        if (shouldSkipSyncingGroupMembers(parentGroup)) {
            return;
        }

        TimedOperation operation = new TimedOperation();
        try {
            dc.addGroupMembershipsForGroup(parentGroup, remoteGroups);
        } finally {
            logger.debug(operation.complete("added [ " + remoteGroups.size() + " ] group members for group [ " + parentGroup.getName() + " ]"));
        }
    }

    @Override
    public void deleteGroupMembersForGroup(final Group parentGroup, final Set<String> remoteGroups) throws OperationFailedException {
        if (shouldSkipSyncingGroupMembers(parentGroup)) {
            return;
        }

        TimedOperation operation = new TimedOperation();
        try {
            dc.removeGroupMembershipsForGroup(parentGroup, remoteGroups);
        } finally {
            logger.debug(operation.complete("removed [ " + remoteGroups.size() + " ] group members for group [ " + parentGroup.getName() + " ]"));
        }
    }

    @Override
    public void deleteCachedGroups(Set<String> groupnames) throws OperationFailedException {
        dc.deleteCachedGroups(groupnames);
    }

    @Override
    public void deleteCachedGroupsByGuids(Set<String> guids) throws OperationFailedException {
        dc.deleteCachedGroupsByGuids(guids);
    }

    @Override
    public void deleteCachedUsersNotIn(Collection<? extends User> users, Date syncStartDate) throws OperationFailedException {
        dc.deleteCachedUsersNotIn(users, syncStartDate);
    }

    // -----------------------------------------------------------------------------------------------------------------
    // Event operations
    // -----------------------------------------------------------------------------------------------------------------

    @Override
    public void addOrUpdateCachedUser(User user) throws OperationFailedException {
        dc.addOrUpdateCachedUser(user);
    }

    @Override
    public void deleteCachedUser(String username) throws OperationFailedException {
        dc.deleteCachedUser(username);
    }

    @Override
    public void addOrUpdateCachedGroup(Group group) throws OperationFailedException {
        dc.addOrUpdateCachedGroup(group);
    }

    @Override
    public void deleteCachedGroup(String groupName) throws OperationFailedException {
        dc.deleteCachedGroup(groupName);
    }

    @Override
    public void addUserToGroup(String username, String groupName) throws OperationFailedException {
        dc.addUserToGroup(username, groupName);
    }

    @Override
    public void removeUserFromGroup(String username, String groupName) throws OperationFailedException {
        dc.removeUserFromGroup(username, groupName);
    }

    @Override
    public void addGroupToGroup(String childGroup, String parentGroup) throws OperationFailedException {
        dc.addGroupToGroup(childGroup, parentGroup);
    }

    @Override
    public void removeGroupFromGroup(String childGroup, String parentGroup) throws OperationFailedException {
        dc.removeGroupFromGroup(childGroup, parentGroup);
    }

    @Override
    public void syncGroupMembershipsForUser(String childUsername, Set<String> parentGroupNames) throws OperationFailedException {
        dc.syncGroupMembershipsForUser(childUsername, parentGroupNames);
    }

    @Override
    public void syncGroupMembershipsAndMembersForGroup(String groupName, Set<String> parentGroupNames, Set<String> childGroupNames) throws OperationFailedException {
        dc.syncGroupMembershipsAndMembersForGroup(groupName, parentGroupNames, childGroupNames);
    }

    @Override
    public Set<String> getAllUserGuids() throws OperationFailedException {
        return dc.getAllUserGuids();
    }

    @Override
    public Set<String> getAllGroupGuids() throws OperationFailedException {
        return dc.getAllGroupGuids();
    }

    @Override
    public Set<String> getAllLocalGroupNames() throws OperationFailedException {
        return dc.getAllLocalGroupNames();
    }

    @Override
    public long getUserCount() throws OperationFailedException {
        return dc.getUserCount();
    }

    @Override
    public long getGroupCount() throws OperationFailedException {
        return dc.getGroupCount();
    }

    @Override
    public long getExternalCachedGroupCount() throws OperationFailedException {
        return dc.getExternalCachedGroupCount();
    }

    @Override
    public UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException {
        return dc.findUserWithAttributesByName(name);
    }

    @Override
    public Map<String, String> findUsersByExternalIds(final Set<String> externalIds) {
        return dc.findUsersByExternalIds(externalIds);
    }

    @Override
    public Map<String, String> findGroupsByExternalIds(final Set<String> externalIds) throws OperationFailedException {
        return dc.findGroupsByExternalIds(externalIds);
    }

    @Override
    public Map<String, String> findGroupsExternalIdsByNames(Set<String> groupNames) throws OperationFailedException {
        return dc.findGroupsExternalIdsByNames(groupNames);
    }

    @Override
    public GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException {
        return dc.findGroupWithAttributesByName(name);
    }

    @Override
    public void applySyncingUserAttributes(String userName,
                                           Set<String> deletedAttributes,
                                           Map<String, Set<String>> storedAttributes)
            throws UserNotFoundException, OperationFailedException {
        dc.applySyncingUserAttributes(userName, deletedAttributes, storedAttributes);
    }

    @Override
    public void applySyncingGroupAttributes(String groupName,
                                            Set<String> deletedAttributes,
                                            Map<String, Set<String>> storedAttributes)
            throws GroupNotFoundException, OperationFailedException {
        dc.applySyncingGroupAttributes(groupName, deletedAttributes, storedAttributes);
    }
}
