package com.atlassian.crowd.directory;

import com.atlassian.crowd.core.event.MultiEventPublisher;
import com.atlassian.crowd.directory.synchronisation.cache.GroupActionStrategy;
import com.atlassian.crowd.directory.synchronisation.utils.AddUpdateSets;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationInformation;
import com.atlassian.crowd.embedded.api.DirectorySynchronisationRoundInformation;
import com.atlassian.crowd.embedded.api.DirectoryType;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.impl.IdentifierMap;
import com.atlassian.crowd.embedded.impl.IdentifierSet;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.embedded.spi.GroupDao;
import com.atlassian.crowd.embedded.spi.UserDao;
import com.atlassian.crowd.event.DirectoryEvent;
import com.atlassian.crowd.event.azure.AzureGroupsRemovedEvent;
import com.atlassian.crowd.event.group.GroupCreatedEvent;
import com.atlassian.crowd.event.group.GroupDeletedEvent;
import com.atlassian.crowd.event.group.GroupMembershipCreatedEvent;
import com.atlassian.crowd.event.group.GroupMembershipDeletedEvent;
import com.atlassian.crowd.event.group.GroupMembershipsCreatedEvent;
import com.atlassian.crowd.event.group.GroupUpdatedEvent;
import com.atlassian.crowd.event.user.UserCreatedFromDirectorySynchronisationEvent;
import com.atlassian.crowd.event.user.UserDeletedEvent;
import com.atlassian.crowd.event.user.UserEditedEvent;
import com.atlassian.crowd.event.user.UserRenamedEvent;
import com.atlassian.crowd.event.user.UsersDeletedEvent;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.directory.SynchronisationStatusManager;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.directory.DirectoryImpl;
import com.atlassian.crowd.model.directory.ImmutableDirectory;
import com.atlassian.crowd.model.directory.SynchronisationStatusKey;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.InternalDirectoryGroup;
import com.atlassian.crowd.model.membership.MembershipType;
import com.atlassian.crowd.model.user.ImmutableUser;
import com.atlassian.crowd.model.user.TimestampedUser;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithCredentialAndAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.Combine;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestriction;
import com.atlassian.crowd.search.query.entity.restriction.BooleanRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.MatchMode;
import com.atlassian.crowd.search.query.entity.restriction.NullRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.TermRestriction;
import com.atlassian.crowd.search.query.entity.restriction.constants.GroupTermKeys;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.atlassian.crowd.util.BatchResult;
import com.atlassian.crowd.util.TimedOperation;
import com.atlassian.crowd.util.TimedProgressOperation;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.atlassian.crowd.attribute.AttributePredicates.SYNCHRONISABLE_ATTRIBUTE_ENTRY_PREDICATE;
import static com.atlassian.crowd.attribute.AttributePredicates.SYNCING_ATTRIBUTE;
import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;
import static com.atlassian.crowd.util.EqualityUtil.different;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;

public class DbCachingRemoteChangeOperations implements DirectoryCacheChangeOperations {
    private static final Logger logger = LoggerFactory.getLogger(DbCachingRemoteChangeOperations.class);

    private final DirectoryDao directoryDao;
    private final RemoteDirectory remoteDirectory;
    private final InternalRemoteDirectory internalDirectory;
    private final SynchronisationStatusManager synchronisationStatusManager;
    private final MultiEventPublisher eventPublisher;
    private final UserDao userDao;
    private final GroupDao groupDao;
    private final GroupActionStrategy groupActionStrategy;

    public DbCachingRemoteChangeOperations(DirectoryDao directoryDao,
                                           RemoteDirectory remoteDirectory,
                                           InternalRemoteDirectory internalDirectory,
                                           SynchronisationStatusManager synchronisationStatusManager,
                                           MultiEventPublisher eventPublisher,
                                           UserDao userDao,
                                           GroupDao groupDao,
                                           GroupActionStrategy groupActionStrategy) {
        this.directoryDao = directoryDao;
        this.remoteDirectory = remoteDirectory;
        this.internalDirectory = internalDirectory;
        this.synchronisationStatusManager = synchronisationStatusManager;
        this.eventPublisher = eventPublisher;
        this.userDao = userDao;
        this.groupDao = groupDao;
        this.groupActionStrategy = groupActionStrategy;
    }

    /**
     * Returns a Map (username -> user) of the internal users created and updated before the specified date.
     *
     * @param date date and time that the user must be updated before to be included in the result.
     *             if {@code null} no restriction will be added all users within the directory will
     *             be returned
     * @return a Map of internal users created and updated before the specified date
     * @throws OperationFailedException if the search operation failed for any reason
     */
    private Map<String, TimestampedUser> findInternalUsersUpdatedBefore(@Nullable Date date) throws OperationFailedException {
        SearchRestriction restriction = date == null ?
                NullRestrictionImpl.INSTANCE :
                Combine.allOf(
                        Restriction.on(UserTermKeys.CREATED_DATE).lessThan(date),
                        Restriction.on(UserTermKeys.UPDATED_DATE).lessThan(date));

        List<TimestampedUser> list = internalDirectory.searchUsers(
                QueryBuilder.queryFor(TimestampedUser.class, EntityDescriptor.user())
                        .with(restriction)
                        .returningAtMost(EntityQuery.ALL_RESULTS));

        Map<String, TimestampedUser> users = new IdentifierMap<>(list.size());
        for (TimestampedUser timestampedUser : list)
            users.put(timestampedUser.getName(), timestampedUser);

        return Collections.unmodifiableMap(users);
    }

    /**
     * Returns a Map (group name -> group) of group created an updated before the specified date.
     *
     * @param date date and time that the group must be updated before to be included in the result
     * @return a Map of groups created and updated before the specified date
     * @throws OperationFailedException if the search operation failed for any reason
     */
    private Map<String, InternalDirectoryGroup> findAndMapByNameGroupsUpdatedBefore(final Date date) throws OperationFailedException {
        final List<InternalDirectoryGroup> groups = findGroupsUpdatedBefore(date);

        return mapGroupsByName(groups);
    }

    private Map<String, InternalDirectoryGroup> mapGroupsByName(final List<InternalDirectoryGroup> groups) {
        final Map<String, InternalDirectoryGroup> result = new IdentifierMap<>(groups.size());
        for (InternalDirectoryGroup internalGroup : groups)
            result.put(internalGroup.getName(), internalGroup);

        return result;
    }

    /**
     * Returns a Map (group external id -> group) of group created an updated before the specified date.
     *
     * @param date date and time that the group must be updated before to be included in the result
     * @return a Map of groups created and updated before the specified date
     * @throws OperationFailedException if the search operation failed for any reason
     */
    private Map<String, InternalDirectoryGroup> findAndMapByExternalIdGroupsUpdatedBefore(final Date date) throws OperationFailedException {
        final List<InternalDirectoryGroup> groups = findGroupsUpdatedBefore(date);

        return mapGroupsByExternalId(groups);
    }

    private Map<String, InternalDirectoryGroup> mapGroupsByExternalId(final List<InternalDirectoryGroup> groups) {
        final Map<String, InternalDirectoryGroup> result = new HashMap<>(groups.size());
        for (InternalDirectoryGroup internalGroup : groups)
            result.put(internalGroup.getExternalId(), internalGroup);

        return result;
    }

    private List<InternalDirectoryGroup> findGroupsUpdatedBefore(final Date date) throws OperationFailedException {
        SearchRestriction restriction = date == null ?
                NullRestrictionImpl.INSTANCE :
                Combine.allOf(
                        Restriction.on(GroupTermKeys.CREATED_DATE).lessThan(date),
                        Restriction.on(GroupTermKeys.UPDATED_DATE).lessThan(date));

        List<InternalDirectoryGroup> groups = internalDirectory.searchGroups(QueryBuilder
                .queryFor(InternalDirectoryGroup.class, EntityDescriptor.group())
                .with(restriction)
                .returningAtMost(EntityQuery.ALL_RESULTS)
        );

        List<InternalDirectoryGroup> roles = internalDirectory.searchGroups(QueryBuilder
                .queryFor(InternalDirectoryGroup.class, EntityDescriptor.role())
                .with(restriction)
                .returningAtMost(EntityQuery.ALL_RESULTS)
        );
        return ImmutableList.<InternalDirectoryGroup>builder().addAll(groups).addAll(roles).build();
    }

    // -----------------------------------------------------------------------------------------------------------------
    // Implements DirectoryCache
    // -----------------------------------------------------------------------------------------------------------------


    @Override
    public void addUsers(Set<UserTemplateWithCredentialAndAttributes> usersToAdd) throws OperationFailedException {
        if (!usersToAdd.isEmpty()) {
            synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.ADDING_USERS, ImmutableList.of(usersToAdd.size()));
            logger.info("adding [ {} ] users", usersToAdd.size());
            TimedOperation operation = new TimedOperation();

            try {
                Directory directory = getDirectory();
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                final BatchResult<User> result = internalDirectory.addAllUsers(usersToAdd);

                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                publishEvents(result.getSuccessfulEntities().stream().map(addedUser -> new UserCreatedFromDirectorySynchronisationEvent(this, immutableDirectory, addedUser)), initialSyncHasBeenStarted);

                logFailures(internalDirectory, result);
                logger.info(operation.complete("added [ " + result.getTotalSuccessful() + " ] users successfully"));
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(operation.complete("failed while adding users"), e);
            }
        }
    }

    @Override
    public void updateUsers(Collection<UserTemplate> usersToUpdate) throws OperationFailedException {
        if (!usersToUpdate.isEmpty()) {
            synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.UPDATING_USERS, ImmutableList.of(usersToUpdate.size()));
            logger.info("updating [ {} ] users", usersToUpdate.size());
            TimedProgressOperation operation = new TimedProgressOperation("updating users", usersToUpdate.size(), logger);
            int successfulUpdates = 0;

            try {
                Directory directory = getDirectory();
                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                for (UserTemplate user : usersToUpdate) {
                    operation.incrementProgress();

                    try {
                        final User originalUser;
                        final String externalId = user.getExternalId();
                        if (StringUtils.isNotEmpty(externalId)) {
                            final User userByExternalId = userByExternalIdOrNull(externalId);
                            if (userByExternalId != null) {
                                originalUser = ImmutableUser.from(userByExternalId);
                                if (!StringUtils.equals(userByExternalId.getName(), user.getName())) {
                                    // CWD-3369: We want to do a rename, but it is possible that first we need to move
                                    // another user out of our way.
                                    final String oldName = userByExternalId.getName();
                                    internalDirectory.forceRenameUser(userByExternalId, user.getName());
                                    publishEvent(new UserRenamedEvent(this, immutableDirectory, user, oldName), initialSyncHasBeenStarted);
                                }
                            } else {
                                // user was not found by externalId, probably we are updating the external id for a directory that didn't have it
                                originalUser = ImmutableUser.from(internalDirectory.findUserByName(user.getName()));
                            }
                        } else {
                            originalUser = ImmutableUser.from(internalDirectory.findUserByName(user.getName()));
                        }

                        final User updatedUser = internalDirectory.updateUser(user);

                        publishEvent(new UserEditedEvent(this, immutableDirectory, updatedUser, originalUser), initialSyncHasBeenStarted);
                        successfulUpdates++;
                    } catch (InvalidUserException e) {
                        // Make sure that one bogus user coming over the wire doesn't hose the entire sync.
                        logger.warn("Unable to synchronize user " + user.getName() + " from remote directory: " + e.getMessage(), e);
                    } catch (UserNotFoundException e) {
                        logger.warn("Could not find user to " + user.getName() + " in internal directory: " + e.getMessage(), e);
                    }
                }
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(operation.complete("failed while updating users"), e);
            } finally {
                logger.info(operation.complete("updated [ " + successfulUpdates + " ] users successfully"));
            }
        }
    }

    @Override
    public void deleteCachedUsersByGuid(Set<String> guids) throws OperationFailedException {
        Set<String> userNamesToDelete = collectUserNamesOfUsersToDeleteByGuid(guids);
        deleteCachedUsersByName(userNamesToDelete);
    }

    @Nullable
    private User userByExternalIdOrNull(final String externalId) {
        try {
            return internalDirectory.findUserByExternalId(externalId);
        } catch (UserNotFoundException e) {
            return null;
        }
    }


    private void deleteCachedUsersByName(Set<String> usernames) throws OperationFailedException {
        synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.DELETING_USERS, ImmutableList.of(usernames.size()));
        logger.info("deleting [ {} ] users", usernames.size());
        TimedOperation operation = new TimedOperation();

        try {
            internalDirectory.removeAllUsers(usernames);

            Directory directory = getDirectory();
            boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

            final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
            publishEvent(new UsersDeletedEvent(this, immutableDirectory, Collections.unmodifiableCollection(usernames)), initialSyncHasBeenStarted);
            // legacy event
            publishEvents(usernames.stream().map(deletedUser -> new UserDeletedEvent(this, immutableDirectory, deletedUser)), initialSyncHasBeenStarted);
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e.getCause());
        } finally {
            logger.info(operation.complete("deleted [ " + usernames.size() + " ] users"));
        }
    }

    private Set<String> collectUserNamesOfUsersToDeleteByGuid(Set<String> guids) {
        final TimedProgressOperation timedProgressOperation = new TimedProgressOperation("Collecting usernames from guids",
                guids.size(), logger);
        final ImmutableSet<String> usernames = ImmutableSet.copyOf(filter(transform(guids, guid -> {
            try {
                final TimestampedUser user = internalDirectory.findUserByExternalId(guid);
                if (user.isMarkedAsDeleted()) {
                    logger.debug("Skipping deletion of user {} from directory {}, because user is already marked as deleted",
                            user.getName(), internalDirectory.getDirectoryId());
                    return null;
                } else {
                    return user.getName();
                }
            } catch (UserNotFoundException e) {
                // It's possible to get deleted guids for users which are not visible to internal directory
                // because of some specified filters.
                logger.debug("user with externalId [ {} ] was not found in [ {} ] and could not be deleted",
                        guid, internalDirectory.getDirectoryId());
                return null;
            } finally {
                timedProgressOperation.incrementedProgress();
            }
        }), Predicates.notNull()));

        timedProgressOperation.complete(String.format("Finished collecting usernames for [ %d ] guids", guids.size()));

        return usernames;
    }

    @Override
    public void deleteCachedUsersNotIn(final Collection<? extends User> remoteUsers, final Date synchStartDate) throws OperationFailedException {
        TimedOperation scanningComparingAndDeletingOperation = new TimedOperation();
        try {
            // Create a HashSet of REMOTE usernames for easy lookup
            Set<String> remoteUsernames = new IdentifierSet(remoteUsers.size());
            Set<String> usersToDelete = new HashSet<>();

            TimedOperation scanningAndComparingOperation = new TimedOperation();
            try {
                for (User remoteUser : remoteUsers) {
                    remoteUsernames.add(remoteUser.getName());
                }

                Set<String> remoteExternalIds = externalIdsOf(remoteUsers);

                Map<String, TimestampedUser> internalUsers = findInternalUsersUpdatedBefore(synchStartDate);

                // Find all Users in our internal cache.

                for (TimestampedUser internalUser : internalUsers.values()) {
                    String userName = internalUser.getName();
                    boolean shouldDelete = StringUtils.isBlank(internalUser.getExternalId()) ?
                            !remoteUsernames.contains(userName) : !remoteExternalIds.contains(internalUser.getExternalId());
                    if (shouldDelete) {
                        if (internalUser.isMarkedAsDeleted()) {
                            logger.debug("user [ {} ] already marked as deleted", userName);
                        } else {
                            logger.debug("user [ {} ] not found, deleting", userName);
                            usersToDelete.add(userName);
                        }
                    }
                }
            } finally {
                logger.info(scanningAndComparingOperation.complete("scanned and compared [ " + remoteUsers.size() + " ] users for delete in DB cache"));
            }


            if (!usersToDelete.isEmpty()) {
                deleteCachedUsersByName(usersToDelete);
            }
        } finally {
            logger.info(scanningComparingAndDeletingOperation.complete("scanned for deleted users"));
        }
    }

    @Override
    public GroupsToAddUpdateReplace findGroupsToUpdate(final Collection<? extends Group> remoteGroups, final Date syncStartDate) throws OperationFailedException {
        ImmutableSet.Builder<GroupTemplate> groupsToAdd = ImmutableSet.builder();
        ImmutableSet.Builder<GroupTemplate> groupsToUpdate = ImmutableSet.builder();
        ImmutableMap.Builder<String, GroupTemplate> groupsToReplace = ImmutableMap.builder();

        TimedOperation operation = new TimedOperation();
        try {
            final List<InternalDirectoryGroup> groupsUpdatedBefore = findGroupsUpdatedBefore(syncStartDate);
            Map<String, InternalDirectoryGroup> groupsByName = mapGroupsByName(groupsUpdatedBefore);
            Map<String, InternalDirectoryGroup> groupsByExternalId = mapGroupsByExternalId(groupsUpdatedBefore);

            for (Group remoteGroup : remoteGroups) {
                InternalDirectoryGroup internalGroup = groupsByName.get(remoteGroup.getName());
                final GroupsToAddUpdateReplace groupsToAddUpdateReplace = groupActionStrategy.decide(internalGroup,
                        groupsByExternalId.get(remoteGroup.getExternalId()), remoteGroup, syncStartDate, getDirectoryId());
                groupsToAdd.addAll(groupsToAddUpdateReplace.groupsToAdd);
                groupsToUpdate.addAll(groupsToAddUpdateReplace.groupsToUpdate);
                groupsToReplace.putAll(groupsToAddUpdateReplace.groupsToReplace);
            }

            return new GroupsToAddUpdateReplace(groupsToAdd.build(), groupsToUpdate.build(), groupsToReplace.build());
        } finally {
            logger.info(operation.complete("scanned and compared [ " + remoteGroups.size() + " ] groups for update in DB cache"));
        }
    }

    @Override
    public void removeGroups(Collection<String> groupsToRemove) throws OperationFailedException {
        if (!groupsToRemove.isEmpty()) {
            TimedOperation operation = new TimedOperation();
            int successfulRemoves = 0;

            try {
                Directory directory = getDirectory();
                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                for (String entry : groupsToRemove) {
                    try {
                        internalDirectory.removeGroup(entry);

                        publishEvent(new GroupDeletedEvent(this, immutableDirectory, entry), initialSyncHasBeenStarted);
                        successfulRemoves++;
                    } catch (GroupNotFoundException e) {
                        logger.warn("Could not find group: " + e.getGroupName(), e);
                    } catch (ReadOnlyGroupException e) {
                        logger.warn("Group is read-only and not allowed to be modified: " + e.getGroupName(), e);
                    }
                }
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(operation.complete("failed while removing groups"), e);
            } finally {
                logger.info(operation.complete("deleted [ " + successfulRemoves + " ] groups to be replaced"));
            }
        }
    }

    @Override
    public void addGroups(Set<GroupTemplate> groupsToAdd) throws OperationFailedException {
        logger.debug("adding [ {} ] groups", groupsToAdd.size());
        if (!groupsToAdd.isEmpty()) {
            synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.ADDING_GROUPS, ImmutableList.of(groupsToAdd.size()));
            TimedOperation operation = new TimedOperation();
            try {
                Directory directory = getDirectory();
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                final BatchResult<Group> result = internalDirectory.addAllGroups(groupsToAdd);

                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                publishEvents(result.getSuccessfulEntities().stream().map(addedGroup -> new GroupCreatedEvent(this, immutableDirectory, addedGroup)), initialSyncHasBeenStarted);
                logFailures(internalDirectory, result);
                logger.info(operation.complete("added [ " + result.getTotalSuccessful() + " ] groups successfully"));
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(operation.complete("failed while adding groups"), e);
            }
        }
    }

    @Override
    public void updateGroups(Collection<GroupTemplate> groupsToUpdate) throws OperationFailedException {
        logger.debug("updating [ {} ] groups", groupsToUpdate.size());
        if (!groupsToUpdate.isEmpty()) {
            synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.UPDATING_GROUPS, ImmutableList.of(groupsToUpdate.size()));
            TimedOperation operation = new TimedOperation();
            int successfulUpdates = 0;

            try {
                Directory directory = getDirectory();
                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                for (GroupTemplate groupTemplate : groupsToUpdate) {
                    try {
                        final Group updatedGroup = internalDirectory.updateGroup(groupTemplate);

                        publishEvent(new GroupUpdatedEvent(this, immutableDirectory, updatedGroup), initialSyncHasBeenStarted);
                        successfulUpdates++;
                    } catch (InvalidGroupException e) {
                        logger.warn("Unable to synchronise group " + groupTemplate.getName() + " with remote directory: " + e.getMessage(), e);
                    } catch (ReadOnlyGroupException e) {
                        logger.warn("Unable to update read-only group " + groupTemplate.getName() + " with remote directory: " + e.getMessage(), e);
                    } catch (GroupNotFoundException e) {
                        // we have just checked that the group exists so if group is not found, something is wrong
                        logger.warn("Unable to find group " + groupTemplate.getName() + " on update with remote directory: " + e.getMessage(), e);
                    }
                }
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(operation.complete("failed while updating groups"), e);
            } finally {
                logger.info(operation.complete("updated [ " + successfulUpdates + " ] groups successfully"));
            }
        }
    }

    @Override
    public void deleteCachedGroupsNotIn(final GroupType groupType, final List<? extends Group> remoteGroups, final Date syncStartDate) throws OperationFailedException {
        final Set<String> groupsToRemove = determineGroupsToRemoveByName(remoteGroups, syncStartDate);

        if (!groupsToRemove.isEmpty()) {
            deleteCachedGroups(groupsToRemove);
        }
    }

    @Override
    public void deleteCachedGroupsNotInByExternalId(final Collection<? extends Group> remoteGroups, final Date syncStartDate) throws OperationFailedException {
        final Set<String> groupsToRemove = determineGroupsToRemoveByExternalId(remoteGroups, syncStartDate);

        if (!groupsToRemove.isEmpty()) {
            deleteCachedGroupsByGuids(groupsToRemove);
            cleanUpAzureAdGroupFiltersIfAny(groupsToRemove);
        }
    }

    private Set<String> determineGroupsToRemoveByName(final List<? extends Group> remoteGroups, Date syncStartDate) throws OperationFailedException {
        TimedOperation operation = new TimedOperation();

        try {
            // Create a HashSet of REMOTE group identifiers for easy lookup
            Set<String> remoteGroupIdentifiers = new IdentifierSet(remoteGroups.size());
            remoteGroupIdentifiers.addAll(remoteGroups.stream().map(Group::getName).collect(Collectors.toSet()));

            Map<String, InternalDirectoryGroup> groups = findAndMapByNameGroupsUpdatedBefore(syncStartDate);
            return processGroups(syncStartDate, remoteGroupIdentifiers, groups, Group::getName);
        } finally {
            logger.info(operation.complete("scanned and compared [ " + remoteGroups.size() + " ] groups for delete in DB cache"));
        }
    }

    private Set<String> determineGroupsToRemoveByExternalId(final Collection<? extends Group> remoteGroups, final Date syncStartDate) throws OperationFailedException {
        TimedOperation operation = new TimedOperation();

        try {
            // Create a HashSet of REMOTE group identifiers for easy lookup
            final Set<String> remoteGroupIdentifiers = ImmutableSet.copyOf(remoteGroups.stream().map(Group::getExternalId).collect(Collectors.toSet()));

            Map<String, InternalDirectoryGroup> groups = findAndMapByExternalIdGroupsUpdatedBefore(syncStartDate);
            return processGroups(syncStartDate, remoteGroupIdentifiers, groups, Group::getExternalId);
        } finally {
            logger.info(operation.complete("scanned and compared [ " + remoteGroups.size() + " ] groups for delete in DB cache"));
        }
    }

    private Set<String> processGroups(Date syncStartDate,
                                      Set<String> remoteGroupIdentifiers,
                                      Map<String, InternalDirectoryGroup> groups,
                                      Function<InternalDirectoryGroup, String> groupIdentifierExtractor) {
        Set<String> groupsToRemove = new HashSet<>();
        for (InternalDirectoryGroup internalGroup : groups.values()) {
            if (internalGroup.isLocal()) {
                continue;
            }
            if (internalGroup.getCreatedDate() == null) {
                logger.warn("group [ " + groupIdentifierExtractor.apply(internalGroup) + " ] in directory [ " + getDirectoryId() + " ] has no created date, skipping");
                // ALWAYS do this comparison with the real millis as these may be SQL Timestamps which will throw away millis whenever they feel like it.
            } else if (syncStartDate != null && internalGroup.getCreatedDate().getTime() > syncStartDate.getTime()) {
                // Don't remove this group, it was added locally after we started our search.
                // Any anomalies will catch up on the next synchronization.
                logger.debug("group [ " + groupIdentifierExtractor.apply(internalGroup) + " ] created after synchronisation start, skipping");
                continue;
            }
            if (!remoteGroupIdentifiers.contains(groupIdentifierExtractor.apply(internalGroup))) {
                logger.debug("group [ " + groupIdentifierExtractor.apply(internalGroup) + " ] not found, deleting");
                groupsToRemove.add(groupIdentifierExtractor.apply(internalGroup));
            }
        }
        return groupsToRemove;
    }

    @Override
    public void deleteCachedGroups(Set<String> groupnames) throws OperationFailedException {
        synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.DELETING_GROUPS, ImmutableList.of(groupnames.size()));
        logger.info("removing [ " + groupnames.size() + " ] groups");
        TimedOperation operation = new TimedOperation();
        try {
            BatchResult<String> result = internalDirectory.removeAllGroups(groupnames);

            Directory directory = getDirectory();
            boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);


            final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
            publishEvents(groupnames.stream().map(groupName -> new GroupDeletedEvent(this, immutableDirectory, groupName)), initialSyncHasBeenStarted);
            logger.info(operation.complete("removed [ " + result.getTotalSuccessful() + " ] groups successfully"));
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(operation.complete("failed while deleting groups"), e);
        }
    }

    public void deleteCachedGroupsByGuids(Set<String> guids) throws OperationFailedException {
        if (!guids.isEmpty()) {
            final List<SearchRestriction> tombstonesAsRestrictions = guids.stream()
                    .map(guid -> new TermRestriction<>(GroupTermKeys.EXTERNAL_ID, MatchMode.EXACTLY_MATCHES, guid)).collect(Collectors.toList());
            final EntityQuery<String> queryForTombstonesNames = QueryBuilder.queryFor(String.class, EntityDescriptor.group())
                    .with(new BooleanRestrictionImpl(BooleanRestriction.BooleanLogic.OR, tombstonesAsRestrictions)).returningAtMost(EntityQuery.ALL_RESULTS);
            final Set<String> groupsToDelete = ImmutableSet.copyOf(internalDirectory.searchGroups(queryForTombstonesNames));
            deleteCachedGroups(groupsToDelete);
            cleanUpAzureAdGroupFiltersIfAny(guids);
        }
    }

    private void cleanUpAzureAdGroupFiltersIfAny(Set<String> groupExternalIdsRemoved) throws OperationFailedException {
        try {
            if (getDirectory().getType() == DirectoryType.AZURE_AD) {
                eventPublisher.publish(new AzureGroupsRemovedEvent(this, getDirectory(), groupExternalIdsRemoved));
            }
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException("failed while finding directory to remove any Azure Filtered Groups", e);
        }
    }

    protected boolean hasChanged(User remoteUser, User internalUser) {
        final boolean externalIdsAreSet = StringUtils.isNotEmpty(remoteUser.getExternalId()) && StringUtils
                .isNotEmpty(internalUser.getExternalId());

        return different(remoteUser.getFirstName(), internalUser.getFirstName()) ||
                different(remoteUser.getLastName(), internalUser.getLastName()) ||
                different(remoteUser.getDisplayName(), internalUser.getDisplayName()) ||
                different(remoteUser.getEmailAddress(), internalUser.getEmailAddress()) ||
                different(remoteUser.getExternalId(), internalUser.getExternalId()) ||
                (externalIdsAreSet && different(remoteUser.getName(), internalUser.getName())) ||
                (remoteDirectory.supportsInactiveAccounts() && remoteUser.isActive() != internalUser.isActive());
    }

    private static UserTemplate makeUserTemplate(User user) {
        UserTemplate template = new UserTemplate(user);
        template.setFirstName(user.getFirstName());
        template.setLastName(user.getLastName());
        template.setDisplayName(user.getDisplayName());
        template.setEmailAddress(user.getEmailAddress());
        return template;
    }

    @Override
    public AddRemoveSets<String> findUserMembershipForGroupChanges(Group group, Collection<String> remoteUsers) throws OperationFailedException {

        Set<String> remoteUsersSet = toLowerCase(remoteUsers);

        TimedOperation operation = new TimedOperation();

        try {

            synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.USER_MEMBERSHIPS, ImmutableList.of(remoteUsersSet.size(), group.getName()));
            logger.debug("synchronising [ " + remoteUsersSet.size() + " ] user members for group [ " + group.getName() + " ]");
            // Remove any internal users from the group if they are not members of the group in REMOTE
            Set<String> internalMembers = toLowerCase(internalDirectory.searchGroupRelationships(QueryBuilder.queryFor(String.class, EntityDescriptor.user()).childrenOf(EntityDescriptor.group()).withName(group.getName()).returningAtMost(EntityQuery.ALL_RESULTS)));

            logger.debug("internal directory has [ " + internalMembers.size() + " ] members");

            Set<String> usersToAdd = Sets.difference(remoteUsersSet, internalMembers);
            Set<String> usersToRemove = Sets.difference(internalMembers, remoteUsersSet);


            return new AddRemoveSets<>(usersToAdd, usersToRemove);
        } finally {
            logger.debug(operation.complete("scanned and compared [ " + remoteUsersSet.size() + " ] user members from [ " + group.getName() + " ]"));
        }
    }

    @Override
    public void removeUserMembershipsForGroup(Group group, Set<String> usersToRemove) throws OperationFailedException {
        if (!usersToRemove.isEmpty()) {
            int failureCount = 0;
            TimedOperation operation = new TimedOperation();
            try {
                Directory directory = getDirectory();
                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                for (String username : usersToRemove) {
                    try {
                        internalDirectory.removeUserFromGroup(username, group.getName());

                        publishEvent(new GroupMembershipDeletedEvent(this, immutableDirectory, username, group.getName(), MembershipType.GROUP_USER), initialSyncHasBeenStarted);
                    } catch (UserNotFoundException e) {
                        failureCount++;
                        logger.debug("Could not remove user [{}] from group [{}]. User was not found in the cache.", username, group.getName());
                    } catch (GroupNotFoundException e) {
                        failureCount++;
                        logger.debug("Could not remove user [{}] from group [{}]. Group was not found in the cache.", username, group.getName());
                    } catch (MembershipNotFoundException e) {
                        // This generally happens when the DAO implementation cascades user deletion to remove
                        // memberships. It's safe to ignore regardless, because the membership not existing is
                        // exactly what we were wanting to happen!
                    } catch (ReadOnlyGroupException e) {
                        failureCount++;
                        logger.warn("Could not remove user [{}] from read-only group [{}].", username, group.getName());
                    }
                }
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(e);
            } finally {
                final int usersRemoved = usersToRemove.size() - failureCount;
                logger.info(operation.complete("removed [ " + usersRemoved + " ] user members from [ " + group.getName() + " ]"));
            }
        }
    }

    @Override
    public void addUserMembershipsForGroup(Group group, Set<String> usersToAdd) throws OperationFailedException {
        if (!usersToAdd.isEmpty()) {
            Collection<String> failedUsernames = null;
            TimedOperation operation = new TimedOperation();
            try {
                Directory directory = getDirectory();
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);

                final BatchResult<String> result = internalDirectory.addAllUsersToGroup(usersToAdd, group.getName());
                failedUsernames = result.getFailedEntities();

                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                publishEvents(result.getSuccessfulEntities().stream().map(username -> new GroupMembershipCreatedEvent(this, immutableDirectory, username, group.getName(), MembershipType.GROUP_USER)),
                        initialSyncHasBeenStarted);
                publishEvent(new GroupMembershipsCreatedEvent(this, immutableDirectory, result.getSuccessfulEntities(), group.getName(), MembershipType.GROUP_USER), initialSyncHasBeenStarted);

                if (!failedUsernames.isEmpty()) {
                    logger.warn("Could not add the following missing users to group [ {} ]: {}", group.getName(),
                            failedUsernames);
                }
            } catch (GroupNotFoundException e) {
                logger.info("Could not add users to group. Group [{}] was not found in the cache. Leaving membership changes for next sync.", e.getGroupName());
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(e);
            } finally {
                final int usersAdded = failedUsernames != null ? usersToAdd.size() - failedUsernames.size() : 0;
                logger.debug(operation.complete("added [ " + usersAdded + " ] user members to [ " + group.getName() + " ]"));
            }
        }
    }

    @Override
    public AddRemoveSets<String> findGroupMembershipForGroupChanges(Group parentGroup, Collection<String> remoteGroups) throws OperationFailedException {
        logger.debug("synchronising [ " + remoteGroups.size() + " ] group members for group [ " + parentGroup.getName() + " ]");

        Set<String> remoteGroupsSet = toLowerCase(remoteGroups);

        TimedOperation operation = new TimedOperation();

        try {
            synchronisationStatusManager.syncStatus(getDirectoryId(), SynchronisationStatusKey.GROUP_MEMBERSHIPS, ImmutableList.of(remoteGroupsSet.size(), parentGroup.getName()));

            Set<String> internalGroups = toLowerCase(internalDirectory.searchGroupRelationships(QueryBuilder.queryFor(String.class, EntityDescriptor.group()).childrenOf(EntityDescriptor.group()).withName(parentGroup.getName()).returningAtMost(EntityQuery.ALL_RESULTS)));

            Set<String> groupsToAdd = Sets.difference(remoteGroupsSet, internalGroups);
            Set<String> groupsToRemove = Sets.difference(internalGroups, remoteGroupsSet);

            return new AddRemoveSets<>(groupsToAdd, groupsToRemove);
        } finally {
            logger.debug(operation.complete("scanned and compared [ " + remoteGroups.size() + " ] group members from [ " + parentGroup.getName() + " ]"));
        }
    }

    @Override
    public void addGroupMembershipsForGroup(Group parentGroup, Collection<String> groupsToAdd) throws OperationFailedException {
        if (!groupsToAdd.isEmpty()) {
            int failureCount = 0;
            TimedOperation operation = new TimedOperation();
            try {
                Directory directory = getDirectory();
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);
                final ImmutableList.Builder<String> addedGroups = ImmutableList.builder();
                for (String groupname : groupsToAdd) {
                    try {
                        internalDirectory.addGroupToGroup(groupname, parentGroup.getName());
                        addedGroups.add(groupname);
                    } catch (GroupNotFoundException e) {
                        failureCount++;
                        logger.info("Could not add child group [{}] to parent group [{}]. Group [{}] was not found in the cache. Leaving membership changes for next sync.",
                                groupname, parentGroup.getName(), e.getGroupName());
                    } catch (InvalidMembershipException e) {
                        failureCount++;
                        logger.warn("Could not add child group [" + groupname + "] to parent group [" + parentGroup.getName() + "]. Membership between child and parent group is invalid", e);
                    } catch (ReadOnlyGroupException e) {
                        failureCount++;
                        logger.warn("Could not add child group [" + groupname + "] to parent group [" + parentGroup.getName() + "]. " + e.getGroupName() + " is a read-only group.", e);
                    } catch (MembershipAlreadyExistsException e) {
                        // The membership already exists in the cache. This happens because the membership was added
                        // during the synchronisation. That's exactly what we want, we don't need to do anything else.
                    }
                }

                final ImmutableList<String> groups = addedGroups.build();
                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                publishEvents(groups.stream().map(groupname -> new GroupMembershipCreatedEvent(this, immutableDirectory, groupname, parentGroup.getName(), MembershipType.GROUP_GROUP)), initialSyncHasBeenStarted);
                publishEvent(new GroupMembershipsCreatedEvent(this, immutableDirectory, groups, parentGroup.getName(), MembershipType.GROUP_GROUP), initialSyncHasBeenStarted);
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(e);
            } finally {
                final int groupsAdded = groupsToAdd.size() - failureCount;
                logger.info(operation.complete("added [ " + groupsAdded + " ] group members to [ " + parentGroup.getName() + " ]"));
            }
        }
    }

    @Override
    public void removeGroupMembershipsForGroup(Group parentGroup, Collection<String> groupsToRemove) throws OperationFailedException {
        if (!groupsToRemove.isEmpty()) {
            int failureCount = 0;
            TimedOperation operation = new TimedOperation();
            try {
                Directory directory = getDirectory();
                boolean initialSyncHasBeenStarted = initialSyncHasBeenStarted(directory);
                final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
                for (String groupname : groupsToRemove) {
                    try {
                        internalDirectory.removeGroupFromGroup(groupname, parentGroup.getName());

                        publishEvent(new GroupMembershipDeletedEvent(this, immutableDirectory, groupname, parentGroup.getName(), MembershipType.GROUP_GROUP), initialSyncHasBeenStarted);
                    } catch (GroupNotFoundException e) {
                        failureCount++;
                        logger.debug("Could not remove child group [{}] from parent group [{}]. Group [{}] was not found. The next sync will fix this problem.", groupname, parentGroup.getName(), e.getGroupName());
                    } catch (InvalidMembershipException e) {
                        failureCount++;
                        logger.warn("Could not remove child group [" + groupname + "] from parent group [" + parentGroup.getName() + "]. Membership between child and parent group is invalid", e);
                    } catch (MembershipNotFoundException e) {
                        // The membership no longer exists in the cache. This happens if it was removed during the
                        // synchronisation. That's exactly what we want, we don't need to do anything else.
                    } catch (ReadOnlyGroupException e) {
                        failureCount++;
                        logger.warn("Could not remove child group [" + groupname + "] from parent group [" + parentGroup.getName() + "]. " + e.getGroupName() + " is a read-only group.", e);
                    }
                }
            } catch (DirectoryNotFoundException e) {
                throw new OperationFailedException(e);
            } finally {
                final int groupsRemoved = groupsToRemove.size() - failureCount;
                logger.debug(operation.complete("removed [ " + groupsRemoved + " ] group members from [ " + parentGroup.getName() + " ]"));
            }
        }
    }

    /**
     * Returns true if the synchronisation
     * has been started at least once after directory creation or configuration
     * update.
     */
    private boolean initialSyncHasBeenStarted(Directory directory) {
        // This method looks awkward for backward compatibility reasons.
        // Old code of this method looked like this:
        //  return directory.getValue(SynchronisableDirectoryProperties.IS_SYNCHRONISING) != null
        // IS_SYNCHRONISING attribute was updated in the following circumstances:
        // - On sync start it was set to true
        // - On sync end it was set to false
        // - On directory update it was set to null, along with setting LAST_CONFIGURATION_CHANGE to now.
        // Effectively it returned true only if there was a sync start or end after directory update.
        // To keep this behaviour we compare highest sync start or end with LAST_CONFIGURATION_CHANGE.
        return getLastSyncStartOrEnd(directory)
                .filter(startOrEnd -> startOrEnd > getLastConfigChangeTimestamp(directory))
                .isPresent();
    }

    private Optional<Long> getLastSyncStartOrEnd(Directory directory) {
        return getLastSyncStartOrEnd(synchronisationStatusManager.getDirectorySynchronisationInformation(directory));
    }

    @VisibleForTesting
    protected static Optional<Long> getLastSyncStartOrEnd(DirectorySynchronisationInformation syncInfo) {
        Optional<Long> activeStart = Optional.ofNullable(syncInfo.getActiveRound())
                .map(DirectorySynchronisationRoundInformation::getStartTime);
        Optional<Long> lastEnd = Optional.ofNullable(syncInfo.getLastRound())
                .map(round -> round.getStartTime() + round.getDurationMs());
        return Collections.max(ImmutableList.of(activeStart, lastEnd),
                Comparator.comparing(o -> o.orElse(Long.MIN_VALUE)));
    }

    private long getLastConfigChangeTimestamp(Directory directory) {
        String value = directory.getValue(DirectoryImpl.LAST_CONFIGURATION_CHANGE);
        return StringUtils.isEmpty(value) ? Long.MIN_VALUE : Long.parseLong(value);
    }

    @VisibleForTesting
    protected Directory getDirectory() throws DirectoryNotFoundException {
        return directoryDao.findById(getDirectoryId());
    }

    private long getDirectoryId() {
        return remoteDirectory.getDirectoryId();
    }

    private void publishEvent(DirectoryEvent event, boolean initialSyncHasBeenStarted) {
        publishEvents(Stream.of(event), initialSyncHasBeenStarted);
    }

    private void publishEvents(Stream<DirectoryEvent> event, boolean initialSyncHasBeenStarted) {
        // Fire event only if this is not initial synchronisation
        if (initialSyncHasBeenStarted) {
            eventPublisher.publishAll(event.collect(Collectors.toList()));
        }
    }

    /**
     * Returns true if the given remote Group should not have its memberships synchronised for any reason.
     *
     * @param remoteGroup The Group to test.
     * @return true if the given remote Group should not have its memberships synchronised for any reason.
     * @throws OperationFailedException If there is an error trying to find the group in the Internal Directory (should not occur).
     */
    @Override
    public GroupShadowingType isGroupShadowed(final Group remoteGroup) throws OperationFailedException {
        try {
            // Find the version of this group in our Internal cache.
            InternalDirectoryGroup internalGroup = internalDirectory.findGroupByName(remoteGroup.getName());
            if (remoteGroup.getType() == GroupType.LEGACY_ROLE && internalGroup.getType() == GroupType.GROUP) {
                return GroupShadowingType.SHADOWED_BY_ROLE;
            } else if (internalGroup.isLocal()) {
                return GroupShadowingType.SHADOWED_BY_LOCAL_GROUP;
            } else {
                return GroupShadowingType.NOT_SHADOWED;
            }
        } catch (GroupNotFoundException ex) {
            // Group does not exist locally - it IS possible someone deleted it while the synchronise is in progress.
            return GroupShadowingType.GROUP_REMOVED;
        }
    }

    /**
     * Returns the users that need to be added or updated given the list of all remote users. Only the internal users
     * modified before <tt>syncStartDate</tt> will be updated. This is done to avoid overriding changes made locally to
     * a user after the synchronisation has started.
     *
     * @param remoteUsers   List of all remote users.
     * @param syncStartDate Date and time of the start of the synchronisation. Used to determine which users need to be
     *                      synchronised. Can be null in which case all the users are synchronised.
     * @return a pair of Sets of users to update and update.
     * @throws OperationFailedException if the operation failed for any reason
     */
    @Override
    public AddUpdateSets<UserTemplateWithCredentialAndAttributes, UserTemplate> getUsersToAddAndUpdate(final Collection<? extends User> remoteUsers, final Date syncStartDate)
            throws OperationFailedException {
        logDuplicates(remoteUsers);
        final ImmutableSet.Builder<UserTemplateWithCredentialAndAttributes> usersToAdd = ImmutableSet.builder();
        final ImmutableSet.Builder<UserTemplate> usersToUpdate = ImmutableSet.builder();
        // Retrieve all internal users, we will filter by updated date in memory that we can correctly identify
        // users which need to be updated and users which need to be created
        final Map<String, TimestampedUser> internalUsersByName = findInternalUsersUpdatedBefore(null);
        final Map<String, TimestampedUser> internalUsersByExternalId = mapUsersByExternalId(internalUsersByName.values());

        // calculate all the external ids of the remote users
        final Set<String> remoteUserExternalIds = externalIdsOf(remoteUsers);

        logger.info("scanning [ {} ] users to add or update", remoteUsers.size());
        TimedProgressOperation operation = new TimedProgressOperation("scanning users to add or update", remoteUsers.size(), logger);

        for (User remoteUser : remoteUsers) {
            operation.incrementProgress();

            TimestampedUser internalUser = null;

            if (StringUtils.isNotEmpty(remoteUser.getExternalId())) {
                internalUser = internalUsersByExternalId.get(remoteUser.getExternalId());
            }

            if (internalUser == null) { // user could not be matched by external Id
                // try matching the user by name, but only if the result wouldn't be matched by external Id with another
                // remote user
                TimestampedUser internalUserMatchedByName = internalUsersByName.get(remoteUser.getName());
                if (internalUserMatchedByName != null &&
                        (StringUtils.isEmpty(internalUserMatchedByName.getExternalId()) ||
                                !remoteUserExternalIds.contains(internalUserMatchedByName.getExternalId()))) {
                    internalUser = internalUserMatchedByName;
                }
            }

            if (internalUser != null) { // remote user was matched to an internal user, so it may be an update
                if (StringUtils.isEmpty(internalUser.getExternalId()) && !remoteUser.getName().equals(internalUser.getName())) {
                    logger.warn("remote username [ {} ] casing differs from local username [ {} ]. User details will be kept updated, but the username cannot be updated",
                            remoteUser.getName(), internalUser.getName());
                }

                if (syncStartDate != null && internalUser.getUpdatedDate() != null &&
                        internalUser.getUpdatedDate().compareTo(syncStartDate) >= 0) {
                    logger.debug("user [ {} ] has been updated since the synchronisation started, skipping", remoteUser.getName());
                } else if (!hasChanged(remoteUser, internalUser)) {
                    logger.trace("user [ {} ] unmodified, skipping", remoteUser.getName());
                } else {
                    final UserTemplate userToUpdate = makeUserTemplate(remoteUser);

                    if (StringUtils.isEmpty(internalUser.getExternalId())) {
                        // Ensure that the username will not be updated (if there's no external id)
                        userToUpdate.setName(internalUser.getName());
                    }

                    // Ignore active flag value from remote directory and manage it locally if the remote directory
                    // does not support active flag or internal directory is maintaining local user status separately
                    if (!remoteDirectory.supportsInactiveAccounts() || internalDirectory.isLocalUserStatusEnabled()) {
                        userToUpdate.setActive(internalUser.isActive()); // local status overrides remote status
                    }

                    usersToUpdate.add(userToUpdate);
                }
            } else { // remote user was not matched to an internal user, so it must be new
                logger.debug("user [ {} ] not found, adding", remoteUser.getName());
                usersToAdd.add(new UserTemplateWithCredentialAndAttributes(makeUserTemplate(remoteUser),
                        PasswordCredential.NONE));
            }
        }
        return new AddUpdateSets<>(usersToAdd.build(), usersToUpdate.build());
    }

    protected static Map<String, TimestampedUser> mapUsersByExternalId(Collection<TimestampedUser> users) {
        final ImmutableMap.Builder<String, TimestampedUser> builder = ImmutableMap.builder();
        for (TimestampedUser user : users) {
            if (StringUtils.isNotEmpty(user.getExternalId())) {
                builder.put(user.getExternalId(), user);
            }
        }
        return builder.build();
    }

    private static Set<String> externalIdsOf(Collection<? extends User> users) {
        final ImmutableSet.Builder<String> builder = ImmutableSet.builder();
        for (User user : users) {
            if (StringUtils.isNotEmpty(user.getExternalId())) {
                builder.add(user.getExternalId());
            }
        }
        return builder.build();
    }

    // Event operations

    @Override
    public void addOrUpdateCachedUser(User user) throws OperationFailedException {
        final UserTemplate newUser = new UserTemplate(user);
        newUser.setDirectoryId(getDirectoryId());
        try {
            final Directory directory = getDirectory();
            final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
            try {
                final User addedUser = internalDirectory.addUser(newUser, PasswordCredential.NONE);

                publishEvent(new UserCreatedFromDirectorySynchronisationEvent(this, immutableDirectory, addedUser), true);
            } catch (UserAlreadyExistsException e) {
                try {
                    final User originalUser = ImmutableUser.from(internalDirectory.findUserByName(newUser.getName()));
                    final User updatedUser = internalDirectory.updateUser(newUser);

                    publishEvent(new UserEditedEvent(this, immutableDirectory, updatedUser, originalUser), true);
                } catch (UserNotFoundException unfe) {
                    // User must have just been deleted locally
                    logger.debug("User was deleted in the middle of the transaction", unfe);
                }
            } catch (InvalidCredentialException e) {
                throw new RuntimeException(e); // Should never happen
            }
        } catch (InvalidUserException e) {
            // Only log the error so that the synchronisation can continue
            logger.error("Could not add or update user '" + newUser.getName() + "'", e);
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public void deleteCachedUser(String username) throws OperationFailedException {
        try {
            internalDirectory.removeUser(username);
            publishEvent(new UsersDeletedEvent(this, getDirectory(), Collections.singleton(username)), true);
            // legacy event
            publishEvent(new UserDeletedEvent(this, getDirectory(), username), true);
        } catch (UserNotFoundException e) {
            logger.debug("Deleted user does not exist locally", e);
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public void addOrUpdateCachedGroup(Group group) throws OperationFailedException {
        final GroupTemplate newGroup = new GroupTemplate(group);
        newGroup.setDirectoryId(getDirectoryId());
        try {
            final Directory directory = getDirectory();
            final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(directory);
            try {
                final Group updatedGroup = internalDirectory.updateGroup(newGroup);

                publishEvent(new GroupUpdatedEvent(this, immutableDirectory, updatedGroup), true);
            } catch (GroupNotFoundException e) {
                final Group addedGroup = internalDirectory.addGroup(newGroup);

                publishEvent(new GroupCreatedEvent(this, immutableDirectory, addedGroup), true);
            } catch (ReadOnlyGroupException e) {
                throw new OperationFailedException(e);
            }
        } catch (InvalidGroupException e) {
            // Only log the error so that the synchronisation can continue
            logger.error("Could not add or update group '" + newGroup.getName() + "'", e);
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public void deleteCachedGroup(String groupName) throws OperationFailedException {
        try {
            internalDirectory.removeGroup(groupName);

            publishEvent(new GroupDeletedEvent(this, getDirectory(), groupName), true);
        } catch (GroupNotFoundException e) {
            logger.debug("Deleted group does not exist locally", e);
        } catch (ReadOnlyGroupException | DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public void addUserToGroup(String username, String groupName) throws OperationFailedException {
        try {
            internalDirectory.addUserToGroup(username, groupName);

            final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(getDirectory());
            publishEvent(new GroupMembershipCreatedEvent(this, immutableDirectory, username, groupName, MembershipType.GROUP_USER), true);
            publishEvent(new GroupMembershipsCreatedEvent(this, immutableDirectory, ImmutableList.of(username), groupName, MembershipType.GROUP_USER), true);
        } catch (GroupNotFoundException e) {
            logger.debug("Cannot have membership without a group", e);
        } catch (UserNotFoundException e) {
            logger.debug("Cannot have membership without a user", e);
        } catch (ReadOnlyGroupException | DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        } catch (MembershipAlreadyExistsException e) {
            logger.debug("The membership specified already exists", e);
        }
    }

    @Override
    public void addGroupToGroup(String childGroup, String parentGroup) throws OperationFailedException {
        try {
            internalDirectory.addGroupToGroup(childGroup, parentGroup);

            final ImmutableDirectory immutableDirectory = ImmutableDirectory.from(getDirectory());
            publishEvent(new GroupMembershipCreatedEvent(this, immutableDirectory, childGroup, parentGroup, MembershipType.GROUP_GROUP), true);
            publishEvent(new GroupMembershipsCreatedEvent(this, immutableDirectory, ImmutableList.of(childGroup), parentGroup, MembershipType.GROUP_GROUP), true);
        } catch (GroupNotFoundException e) {
            logger.debug("Cannot have membership without a group", e);
        } catch (InvalidMembershipException e) {
            logger.debug("Later events should fix this problem", e);
        } catch (ReadOnlyGroupException | DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        } catch (MembershipAlreadyExistsException e) {
            logger.debug("The membership specified already exists", e);
        }
    }

    @Override
    public void removeUserFromGroup(String username, String groupName) throws OperationFailedException {
        try {
            internalDirectory.removeUserFromGroup(username, groupName);

            publishEvent(new GroupMembershipDeletedEvent(this, getDirectory(), username, groupName, MembershipType.GROUP_USER), true);
        } catch (MembershipNotFoundException e) {
            logger.debug("Membership has already been removed", e);
        } catch (GroupNotFoundException e) {
            logger.debug("Cannot have membership without a group", e);
        } catch (UserNotFoundException e) {
            logger.debug("Cannot have membership without a user", e);
        } catch (ReadOnlyGroupException | DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public void removeGroupFromGroup(String childGroup, String parentGroup) throws OperationFailedException {
        try {
            internalDirectory.removeGroupFromGroup(childGroup, parentGroup);

            publishEvent(new GroupMembershipDeletedEvent(this, getDirectory(), childGroup, parentGroup, MembershipType.GROUP_GROUP), true);
        } catch (MembershipNotFoundException e) {
            logger.debug("Membership has already been removed", e);
        } catch (GroupNotFoundException e) {
            logger.debug("Cannot have membership without a group", e);
        } catch (InvalidMembershipException e) {
            logger.debug("Later events should fix this problem", e);
        } catch (ReadOnlyGroupException | DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public void syncGroupMembershipsForUser(String childUsername, Set<String> parentGroupNames) throws OperationFailedException {
        final Set<String> remoteParentGroupNames = toLowerCase(parentGroupNames);
        final Set<String> localParentGroupNames = toLowerCase(internalDirectory.searchGroupRelationships(
                QueryBuilder.queryFor(String.class, EntityDescriptor.group(GroupType.GROUP))
                        .parentsOf(EntityDescriptor.user())
                        .withName(childUsername)
                        .returningAtMost(EntityQuery.ALL_RESULTS)));

        final Set<String> addedParentGroupNames = Sets.difference(remoteParentGroupNames, localParentGroupNames);
        for (String addedParentGroupName : addedParentGroupNames) {
            addUserToGroup(childUsername, addedParentGroupName);
        }
        final Set<String> removedParentGroupNames = Sets.difference(localParentGroupNames, remoteParentGroupNames);
        for (String removedParentGroupName : removedParentGroupNames) {
            removeUserFromGroup(childUsername, removedParentGroupName);
        }
    }

    @Override
    public void syncGroupMembershipsAndMembersForGroup(String groupName, Set<String> parentGroupNames, Set<String> childGroupNames) throws OperationFailedException {
        // Sync memberships
        final Set<String> remoteParentGroupNames = toLowerCase(parentGroupNames);
        final Set<String> localParentGroupNames = toLowerCase(internalDirectory.searchGroupRelationships(
                QueryBuilder.queryFor(String.class, EntityDescriptor.group(GroupType.GROUP))
                        .parentsOf(EntityDescriptor.group(GroupType.GROUP))
                        .withName(groupName)
                        .returningAtMost(EntityQuery.ALL_RESULTS)));

        final Set<String> addedParentGroupNames = Sets.difference(remoteParentGroupNames, localParentGroupNames);
        for (String addedParentGroupName : addedParentGroupNames) {
            addGroupToGroup(groupName, addedParentGroupName);
        }
        final Set<String> removedParentGroupNames = Sets.difference(localParentGroupNames, remoteParentGroupNames);
        for (String removedParentGroupName : removedParentGroupNames) {
            removeGroupFromGroup(groupName, removedParentGroupName);
        }

        // Sync members
        final Set<String> remoteChildGroupNames = toLowerCase(childGroupNames);
        final Set<String> localChildGroupNames = toLowerCase(internalDirectory.searchGroupRelationships(
                QueryBuilder.queryFor(String.class, EntityDescriptor.group(GroupType.GROUP))
                        .childrenOf(EntityDescriptor.group(GroupType.GROUP))
                        .withName(groupName)
                        .returningAtMost(EntityQuery.ALL_RESULTS)));

        final Set<String> addedChildGroupNames = Sets.difference(remoteChildGroupNames, localChildGroupNames);
        for (String addedChildGroupName : addedChildGroupNames) {
            addGroupToGroup(addedChildGroupName, groupName);
        }
        final Set<String> removedChildGroupNames = Sets.difference(localChildGroupNames, remoteChildGroupNames);
        for (String removedChildGroupName : removedChildGroupNames) {
            removeGroupFromGroup(removedChildGroupName, groupName);
        }
    }

    @Override
    public UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException {
        return internalDirectory.findUserWithAttributesByName(name);
    }

    @Override
    public GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException {
        return internalDirectory.findGroupWithAttributesByName(name);
    }

    @Override
    public Map<String, String> findUsersByExternalIds(final Set<String> externalIds) {
        return userDao.findByExternalIds(getDirectoryId(), externalIds);
    }

    @Override
    public Map<String, String> findGroupsByExternalIds(final Set<String> externalIds) throws OperationFailedException {
        return groupDao.findByExternalIds(getDirectoryId(), externalIds);
    }

    @Override
    public Map<String, String> findGroupsExternalIdsByNames(Set<String> groupNames) throws OperationFailedException {
        return groupDao.findExternalIdsByNames(getDirectoryId(), groupNames);
    }

    @Override
    public void applySyncingUserAttributes(String userName, Set<String> deletedAttributes, Map<String, Set<String>> storedAttributes)
            throws UserNotFoundException, OperationFailedException {
        // We handle deletedAttributes and storedAttributes being null here because the values being passed in can come from various places which could potentially allow null,
        // for example from an event that doesn't even care about attributes, we don't want to explicitly say that there is an empty set. It's less that it is empty, and
        // more that it is just not used. In this case, it is acceptable for this method to just ignore them.
        if (deletedAttributes != null) {
            List<String> filteredAttributes = deletedAttributes.stream()
                    .filter(SYNCING_ATTRIBUTE).collect(Collectors.toList());
            for (String key : filteredAttributes) {
                internalDirectory.removeUserAttributes(userName, key);
            }
        }
        if (MapUtils.isNotEmpty(storedAttributes)) {
            Map<String, Set<String>> filteredAttributes = storedAttributes.entrySet().stream()
                    .filter(SYNCHRONISABLE_ATTRIBUTE_ENTRY_PREDICATE)
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            internalDirectory.storeUserAttributes(userName, filteredAttributes);
        }
    }

    @Override
    public void applySyncingGroupAttributes(String groupName, Set<String> deletedAttributes, Map<String, Set<String>> storedAttributes) throws GroupNotFoundException, OperationFailedException {
        // We handle deletedAttributes and storedAttributes being null here because the values being passed in can come from various places which could potentially allow null,
        // for example from a GroupEvent that doesn't even care about attributes, we don't want to explicitly say that there is an empty set. It's less that it is empty, and
        // more that it is just not used. In this case, it is acceptable for this method to just ignore them.
        if (deletedAttributes != null) {
            List<String> filteredAttributes = deletedAttributes.stream()
                    .filter(SYNCING_ATTRIBUTE).collect(Collectors.toList());
            for (String key : filteredAttributes) {
                internalDirectory.removeGroupAttributes(groupName, key);
            }
        }
        if (MapUtils.isNotEmpty(storedAttributes)) {
            Map<String, Set<String>> filteredAttributes = storedAttributes.entrySet().stream()
                    .filter(SYNCHRONISABLE_ATTRIBUTE_ENTRY_PREDICATE)
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            internalDirectory.storeGroupAttributes(groupName, filteredAttributes);
        }
    }

    @Override
    public Set<String> getAllUserGuids() throws OperationFailedException {
        return internalDirectory.getAllUserExternalIds();
    }

    @Override
    public Set<String> getAllGroupGuids() throws OperationFailedException {
        try {
            return groupDao.getAllExternalIds(this.getDirectoryId());
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public long getUserCount() throws OperationFailedException {
        return internalDirectory.getUserCount();
    }

    @Override
    public long getGroupCount() throws OperationFailedException {
        try {
            return groupDao.getGroupCount(this.getDirectoryId());
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public long getExternalCachedGroupCount() throws OperationFailedException {
        try {
            return groupDao.getExternalGroupCount(this.getDirectoryId());
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    @Override
    public Set<String> getAllLocalGroupNames() throws OperationFailedException {
        try {
            return groupDao.getLocalGroupNames(getDirectoryId());
        } catch (DirectoryNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    private static void logFailures(InternalRemoteDirectory directory, final BatchResult<? extends DirectoryEntity> result) {
        if (result.hasFailures()) {
            String directoryName = directory.getDescriptiveName();
            for (DirectoryEntity failedEntity : result.getFailedEntities()) {
                logger.warn("Could not add the following entity to the directory [ {} ]: {}", directoryName, failedEntity.getName());
            }
        }
    }

    private void logDuplicates(final Collection<? extends User> remoteUsers) {
        if (logger.isTraceEnabled()) {
            logger.trace("Starting scanning for not unique users in remote directory: {}.", remoteDirectory.getDirectoryId());
            Set<User> unique = new HashSet<>();
            for (User user : remoteUsers) {
                if (!unique.add(user)) {
                    logger.trace("user [ {}, externalId: {} ] is not unique in remote directory {}.",
                            user.getName(), user.getExternalId(), user.getDirectoryId());
                }
            }
            logger.trace("Completed scanning for not unique users in remote directory: {}.", remoteDirectory.getDirectoryId());
        }
    }
}
