package com.atlassian.crowd.directory;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.crowd.audit.AuditLogEntityType;
import com.atlassian.crowd.audit.AuditLogEntry;
import com.atlassian.crowd.audit.AuditLogEventType;
import com.atlassian.crowd.audit.ImmutableAuditLogChangeset;
import com.atlassian.crowd.audit.ImmutableAuditLogEntity;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.audit.AuditService;
import com.atlassian.crowd.manager.audit.mapper.AuditLogGroupMapper;
import com.atlassian.crowd.manager.audit.mapper.AuditLogUserMapper;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.util.BoundedCount;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * A decorator around a {@link RemoteDirectory} that creates events in the audit log upon changes. Used to extend
 * existing directories with auditing. This class should not be used for directories using Crowd's persistence layer as
 * the changes will be audited by the DAOs.
 *
 * @since 3.2.0
 */
public class AuditingDirectoryDecorator implements RemoteDirectory {

    private final RemoteDirectory remoteDirectory;
    private final AuditService auditService;
    private final ImmutableAuditLogEntity directoryEntity;
    private final AuditLogUserMapper auditLogUserMapper;
    private final AuditLogGroupMapper auditLogGroupMapper;

    public AuditingDirectoryDecorator(
            final RemoteDirectory remoteDirectory,
            final AuditService auditService,
            final AuditLogUserMapper auditLogUserMapper,
            final AuditLogGroupMapper auditLogGroupMapper,
            final String directoryName) {
        this.remoteDirectory = remoteDirectory;
        this.auditService = auditService;
        this.auditLogUserMapper = auditLogUserMapper;
        this.auditLogGroupMapper = auditLogGroupMapper;
        this.directoryEntity = new ImmutableAuditLogEntity.Builder()
                .setEntityId(remoteDirectory.getDirectoryId())
                .setEntityName(directoryName)
                .setEntityType(AuditLogEntityType.DIRECTORY)
                .build();
    }

    @Override
    public void addUserToGroup(String username, String groupName) throws GroupNotFoundException, UserNotFoundException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        remoteDirectory.addUserToGroup(username, groupName);
        auditMembershipEvent(AuditLogEventType.ADDED_TO_GROUP, groupName, username, AuditLogEntityType.USER);
    }

    @Override
    public void addGroupToGroup(String childGroup, String parentGroup) throws GroupNotFoundException, InvalidMembershipException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        remoteDirectory.addGroupToGroup(childGroup, parentGroup);
        auditMembershipEvent(AuditLogEventType.ADDED_TO_GROUP, parentGroup, childGroup, AuditLogEntityType.GROUP);
    }

    @Override
    public void removeUserFromGroup(String username, String groupName) throws GroupNotFoundException, UserNotFoundException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        remoteDirectory.removeUserFromGroup(username, groupName);
        auditMembershipEvent(AuditLogEventType.REMOVED_FROM_GROUP, groupName, username, AuditLogEntityType.USER);
    }

    @Override
    public void removeGroupFromGroup(String childGroup, String parentGroup) throws GroupNotFoundException, InvalidMembershipException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        remoteDirectory.removeGroupFromGroup(childGroup, parentGroup);
        auditMembershipEvent(AuditLogEventType.REMOVED_FROM_GROUP, parentGroup, childGroup, AuditLogEntityType.GROUP);
    }

    @Override
    @Nonnull
    public <T> List<T> searchGroupRelationships(MembershipQuery<T> query) throws OperationFailedException {
        return remoteDirectory.searchGroupRelationships(query);
    }

    @Override
    public void testConnection() throws OperationFailedException {
        remoteDirectory.testConnection();
    }

    @Override
    public boolean supportsInactiveAccounts() {
        return remoteDirectory.supportsInactiveAccounts();
    }

    @Override
    public boolean supportsNestedGroups() {
        return remoteDirectory.supportsNestedGroups();
    }

    @Override
    public boolean supportsPasswordExpiration() {
        return remoteDirectory.supportsPasswordExpiration();
    }

    @Override
    public boolean supportsSettingEncryptedCredential() {
        return remoteDirectory.supportsSettingEncryptedCredential();
    }

    @Override
    public boolean isRolesDisabled() {
        return remoteDirectory.isRolesDisabled();
    }

    @Override
    @Nonnull
    public Iterable<Membership> getMemberships() throws OperationFailedException {
        return remoteDirectory.getMemberships();
    }

    @Override
    @Nonnull
    public RemoteDirectory getAuthoritativeDirectory() {
        return remoteDirectory.getAuthoritativeDirectory();
    }

    @Override
    public void expireAllPasswords() throws OperationFailedException {
        remoteDirectory.expireAllPasswords();
    }

    @Override
    @Nullable
    public AvatarReference getUserAvatarByName(String username, int sizeHint) throws UserNotFoundException, OperationFailedException {
        return remoteDirectory.getUserAvatarByName(username, sizeHint);
    }

    @Override
    @ExperimentalApi
    public User updateUserFromRemoteDirectory(User remoteUser) throws OperationFailedException, UserNotFoundException {
        return remoteDirectory.updateUserFromRemoteDirectory(remoteUser);
    }

    @Override
    @Nullable
    public Set<String> getValues(String key) {
        return remoteDirectory.getValues(key);
    }

    @Override
    @Nullable
    public String getValue(String key) {
        return remoteDirectory.getValue(key);
    }

    @Override
    public Set<String> getKeys() {
        return remoteDirectory.getKeys();
    }

    @Override
    public boolean isEmpty() {
        return remoteDirectory.isEmpty();
    }

    @Override
    @Nonnull
    public User addUser(UserTemplate user, PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        final User addedUser = remoteDirectory.addUser(user, credential);

        auditUserEvent(AuditLogEventType.USER_CREATED, user.getName(), auditLogUserMapper.calculateDifference(AuditLogEventType.USER_CREATED, null, user));
        return addedUser;
    }

    @Override
    public UserWithAttributes addUser(UserTemplateWithAttributes user, PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        final UserWithAttributes addedUser = remoteDirectory.addUser(user, credential);

        auditUserEvent(AuditLogEventType.USER_CREATED, user.getName(), auditLogUserMapper.calculateDifference(AuditLogEventType.USER_CREATED, null, user));
        return addedUser;
    }

    @Nonnull
    @Override
    public User updateUser(UserTemplate user) throws InvalidUserException, UserNotFoundException, OperationFailedException {
        final User oldUser = findUserByName(user.getName());
        final User updatedUser = remoteDirectory.updateUser(user);

        auditUserEvent(AuditLogEventType.USER_UPDATED, user.getName(), auditLogUserMapper.calculateDifference(AuditLogEventType.USER_UPDATED, oldUser, user));
        return updatedUser;
    }

    @Nonnull
    @Override
    public User renameUser(String oldName, String newName) throws UserNotFoundException, InvalidUserException, UserAlreadyExistsException, OperationFailedException {
        final User userToRename = findUserByName(oldName);
        final User renamedUser = remoteDirectory.renameUser(oldName, newName);

        auditUserEvent(AuditLogEventType.USER_UPDATED, newName, auditLogUserMapper.calculateDifference(AuditLogEventType.USER_DELETED, userToRename, renamedUser));
        return renamedUser;
    }

    @Override
    public void storeUserAttributes(String username, Map<String, Set<String>> attributes) throws UserNotFoundException, OperationFailedException {
        remoteDirectory.storeUserAttributes(username, attributes);
    }

    @Override
    public void removeUserAttributes(String username, String attributeName) throws UserNotFoundException, OperationFailedException {
        remoteDirectory.removeUserAttributes(username, attributeName);
    }

    @Override
    public void removeUser(String name) throws UserNotFoundException, OperationFailedException {
        final User userToRemove = findUserByName(name);
        remoteDirectory.removeUser(name);

        auditUserEvent(AuditLogEventType.USER_DELETED, name, auditLogUserMapper.calculateDifference(AuditLogEventType.USER_DELETED, userToRemove, null));
    }

    @Override
    @Nonnull
    public <T> List<T> searchUsers(EntityQuery<T> query) throws OperationFailedException {
        return remoteDirectory.searchUsers(query);
    }

    @Override
    @Nonnull
    public Group findGroupByName(String name) throws GroupNotFoundException, OperationFailedException {
        return remoteDirectory.findGroupByName(name);
    }

    @Override
    @Nonnull
    public GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException {
        return remoteDirectory.findGroupWithAttributesByName(name);
    }

    @Override
    public void updateUserCredential(String username, PasswordCredential credential) throws UserNotFoundException, InvalidCredentialException, OperationFailedException {
        remoteDirectory.updateUserCredential(username, credential);

        auditLogUpdateUserCredential(username);
    }

    @Nonnull
    @Override
    public Group addGroup(GroupTemplate group) throws InvalidGroupException, OperationFailedException {
        final Group addedGroup = remoteDirectory.addGroup(group);
        auditLogGroupOperation(group.getName(), auditLogGroupMapper.calculateDifference(null, addedGroup),
                AuditLogEventType.GROUP_CREATED);
        return addedGroup;
    }

    @Override
    public void removeGroup(String name) throws GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        final Group groupToRemove = remoteDirectory.findGroupByName(name);
        remoteDirectory.removeGroup(name);
        auditLogGroupOperation(name, auditLogGroupMapper.calculateDifference(groupToRemove, null),
                AuditLogEventType.GROUP_DELETED);
    }

    @Override
    @Nonnull
    public <T> List<T> searchGroups(EntityQuery<T> query) throws OperationFailedException {
        return remoteDirectory.searchGroups(query);
    }

    @Override
    public boolean isUserDirectGroupMember(String username, String groupName) throws OperationFailedException {
        return remoteDirectory.isUserDirectGroupMember(username, groupName);
    }

    @Override
    public boolean isGroupDirectGroupMember(String childGroup, String parentGroup) throws OperationFailedException {
        return remoteDirectory.isGroupDirectGroupMember(childGroup, parentGroup);
    }

    @Override
    @Nonnull
    public BoundedCount countDirectMembersOfGroup(String groupName, int querySizeHint) throws OperationFailedException {
        return remoteDirectory.countDirectMembersOfGroup(groupName, querySizeHint);
    }

    @Nonnull
    @Override
    public Group updateGroup(GroupTemplate group) throws InvalidGroupException, GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        final Group groupBeforeUpdate = remoteDirectory.findGroupByName(group.getName());
        final Group updatedGroup = remoteDirectory.updateGroup(group);
        auditLogGroupOperation(group.getName(), auditLogGroupMapper.calculateDifference(groupBeforeUpdate, updatedGroup),
                AuditLogEventType.GROUP_UPDATED);
        return updatedGroup;
    }

    @Override
    @Nonnull
    public Group renameGroup(String oldName, String newName) throws GroupNotFoundException, InvalidGroupException, OperationFailedException {
        return remoteDirectory.renameGroup(oldName, newName);
    }

    @Override
    public void storeGroupAttributes(String groupName, Map<String, Set<String>> attributes) throws GroupNotFoundException, OperationFailedException {
        remoteDirectory.storeGroupAttributes(groupName, attributes);
    }

    @Override
    public void removeGroupAttributes(String groupName, String attributeName) throws GroupNotFoundException, OperationFailedException {
        remoteDirectory.removeGroupAttributes(groupName, attributeName);
    }

    @Override
    public long getDirectoryId() {
        return remoteDirectory.getDirectoryId();
    }

    @Override
    public void setDirectoryId(long directoryId) {
        remoteDirectory.setDirectoryId(directoryId);
    }

    @Override
    @Nonnull
    public String getDescriptiveName() {
        return remoteDirectory.getDescriptiveName();
    }

    @Override
    public void setAttributes(Map<String, String> attributes) {
        remoteDirectory.setAttributes(attributes);
    }

    @Override
    @Nonnull
    public User findUserByName(String name) throws UserNotFoundException, OperationFailedException {
        return remoteDirectory.findUserByName(name);
    }

    @Override
    @Nonnull
    public UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException {
        return remoteDirectory.findUserWithAttributesByName(name);
    }

    @Override
    @Nonnull
    public User findUserByExternalId(String externalId) throws UserNotFoundException, OperationFailedException {
        return remoteDirectory.findUserByExternalId(externalId);
    }

    @Override
    @Nonnull
    public User authenticate(String name, PasswordCredential credential) throws UserNotFoundException, InactiveAccountException, InvalidAuthenticationException, ExpiredCredentialException, OperationFailedException {
        return remoteDirectory.authenticate(name, credential);
    }

    private void auditLogUpdateUserCredential(String username) {
        final ImmutableAuditLogEntity primaryUserEntity = new ImmutableAuditLogEntity.Builder()
                .setPrimary()
                .setEntityName(username)
                .setEntityType(AuditLogEntityType.USER)
                .build();

        final ImmutableAuditLogChangeset auditLogChangeset = new ImmutableAuditLogChangeset.Builder()
                .setEventType(AuditLogEventType.PASSWORD_CHANGED)
                .addEntity(primaryUserEntity)
                .addEntity(directoryEntity)
                .addEntry(auditLogUserMapper.calculatePasswordDiff())
                .build();

        auditService.saveAudit(auditLogChangeset);
    }

    private void auditMembershipEvent(AuditLogEventType eventType, String parentName, String childName, AuditLogEntityType childType) {
        final ImmutableAuditLogEntity parentGroupEntity = new ImmutableAuditLogEntity.Builder()
                .setEntityName(parentName)
                .setEntityType(AuditLogEntityType.GROUP)
                .setPrimary()
                .build();
        final ImmutableAuditLogEntity childEntity = new ImmutableAuditLogEntity.Builder()
                .setEntityName(childName)
                .setEntityType(childType)
                .build();

        auditService.saveAudit(
                new ImmutableAuditLogChangeset.Builder()
                        .setEventType(eventType)
                        .addEntity(parentGroupEntity)
                        .addEntity(childEntity)
                        .addEntity(directoryEntity)
                        .build()
        );
    }

    private void auditUserEvent(AuditLogEventType eventType, String name, List<AuditLogEntry> entries) {
        if (entries.size() == 0) {
            return;
        }

        final ImmutableAuditLogEntity userEntity = new ImmutableAuditLogEntity.Builder()
                .setEntityName(name)
                .setEntityType(AuditLogEntityType.USER)
                .setPrimary()
                .build();

        auditService.saveAudit(
                new ImmutableAuditLogChangeset.Builder()
                        .setEventType(eventType)
                        .addEntity(userEntity)
                        .addEntity(directoryEntity)
                        .addEntries(entries)
                        .build()
        );
    }

    private void auditLogGroupOperation(String groupName, List<AuditLogEntry> diffEntries, AuditLogEventType eventType) {
        if (diffEntries.size() == 0) {
            return;
        }

        final ImmutableAuditLogEntity primaryGroupEntity = new ImmutableAuditLogEntity.Builder()
                .setEntityName(groupName)
                .setEntityType(AuditLogEntityType.GROUP)
                .setPrimary()
                .build();

        final ImmutableAuditLogChangeset auditLogChangeset = new ImmutableAuditLogChangeset.Builder()
                .addEntity(primaryGroupEntity)
                .addEntity(directoryEntity)
                .setEventType(eventType)
                .addEntries(diffEntries)
                .build();

        auditService.saveAudit(auditLogChangeset);
    }
}
