package com.atlassian.crowd.core.event.listener;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.event.user.UserAuthenticatedEvent;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.OperationNotSupportedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.directory.DirectoryPermissionException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.Collection;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;

public class AutoGroupAdderListener {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    private final DirectoryManager directoryManager;
    private final List<DefaultGroupMembershipResolver> defaultMembershipResolvers;
    private final EventPublisher eventPublisher;

    public AutoGroupAdderListener(DirectoryManager directoryManager, List<DefaultGroupMembershipResolver> defaultMembershipResolvers, EventPublisher eventPublisher) {
        this.directoryManager = checkNotNull(directoryManager, "directoryManager");
        this.defaultMembershipResolvers = checkNotNull(defaultMembershipResolvers);
        this.eventPublisher = eventPublisher;
    }

    @PostConstruct
    public void register() {
        eventPublisher.register(this);
    }

    @PreDestroy
    public void unregister() {
        eventPublisher.unregister(this);
    }

    @EventListener
    public void handleEvent(final UserAuthenticatedEvent event) {
        final Directory directory = event.getDirectory();
        final User user = event.getUser();
        final Application application = event.getApplication();
        handleEvent(directory, user, application);
    }

    private void handleEvent(Directory directory, User user, Application application) {
        try {
            final UserWithAttributes userWithAttributes = directoryManager.findUserWithAttributesByName(directory.getId(), user.getName());
            defaultMembershipResolvers.forEach(resolver -> {
                final Collection<String> defaultGroupNames = resolver.getDefaultGroupNames(application, directory, userWithAttributes);
                if (!defaultGroupNames.isEmpty()) {
                    defaultGroupNames.forEach(group -> addUserToGroupSafely(directory, userWithAttributes, group,
                            application));
                    try {
                        resolver.onDefaultGroupsAdded(application, directory, userWithAttributes);
                    } catch (OperationFailedException e) {
                        logger.error("Could not call back resolver", e);
                        throw new RuntimeException(e);
                    }

                }
            });
        } catch (DirectoryInstantiationException e) {
            logger.error("Could not instantiate directory: {}", e.getMessage(), e);
        } catch (OperationFailedException e) {
            logger.error("Could not access directory: {}", e.getMessage(), e);
        } catch (UserNotFoundException e) {
            logger.error("Could not access user: {}", e.getMessage(), e);
        } catch (DirectoryNotFoundException e) {
            logger.error("Could not find directory {}", directory.getId(), e);
        }
    }

    private void addUserToGroupSafely(Directory directory, User user, String groupName, Application application) {
        try {
            directoryManager.addUserToGroup(directory.getId(), user.getName(), groupName);
        } catch (GroupNotFoundException e) {
            logger.warn("Could not auto add user {} to group {}, because the group does not exist. User is authenticating" +
                    "to directory {} from application {}", user.getName(), groupName, directory.getName(), application.getName());
            logger.debug("Underlying exception", e);
        } catch (ReadOnlyGroupException e) {
            logger.error("Could not auto add user {} to group {}, because the group is read only. User is authenticating" +
                    "to directory {} from application {}", user.getName(), groupName, directory.getName(), application.getName());
            logger.debug("Underlying exception", e);
        } catch (UserNotFoundException e) {
            logger.error("Could not auto add user to group: {}", e.getMessage(), e);
        } catch (OperationNotSupportedException e) {
            logger.warn("Could not add user to default groups because directory {}[{}] doesn't support such operation", directory.getName(), directory.getId());
            logger.debug("Underlying exception", e);
        } catch (OperationFailedException e) {
            logger.error("Could not access directory: {}", e.getMessage(), e);
        } catch (DirectoryNotFoundException e) {
            logger.error("Could not find directory {}", directory.getId(), e);
        } catch (DirectoryPermissionException e) {
            logger.error("You have group <{}> to be auto-added for the user <{}>, but the directory does not have permission for Group updates.", groupName, user.getName());
        } catch (MembershipAlreadyExistsException e) {
            logger.debug("Could not auto add user to group because membership already exists", e);
        }
    }
}
