package com.atlassian.crowd.core.event.listener;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.application.DefaultGroupMembershipService;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.directory.DirectoryPermissionException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.application.ApplicationDirectoryMapping;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class ApplicationDefaultGroupMembershipResolver implements DefaultGroupMembershipResolver {
    private static final Logger logger = LoggerFactory.getLogger(ApplicationDefaultGroupMembershipResolver.class);

    private final DefaultGroupMembershipService defaultGroupMembershipService;
    private final DirectoryManager directoryManager;

    public ApplicationDefaultGroupMembershipResolver(DefaultGroupMembershipService defaultGroupMembershipService, DirectoryManager directoryManager) {
        this.defaultGroupMembershipService = defaultGroupMembershipService;
        this.directoryManager = directoryManager;
    }

    @Override
    public Collection<String> getDefaultGroupNames(Application application, Directory directory, UserWithAttributes user) {
        try {
            if (userHasAttributeForApplicationAlreadySet(application, user)) {
                logger.debug("User '{}' won't be added to default groups for application '{}', because the user has been " +
                        "already added to default groups for that application", user.getName(), application.getName());
                return Collections.emptySet();
            }
            final ApplicationDirectoryMapping mapping = application.getApplicationDirectoryMapping(directory.getId());
            // Mapping can be null in recovery mode.
            final List<String> groups = mapping == null
                    ? ImmutableList.of() : defaultGroupMembershipService.listAll(application, mapping);
            if (!groups.isEmpty()) {
                logger.info("User '{}' will be added to default groups of application '{}': {}", user.getName(), application.getName(), groups);
            }
            return groups;
        } catch (OperationFailedException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void onDefaultGroupsAdded(Application application, Directory directory, UserWithAttributes user) throws OperationFailedException {
        try {
            directoryManager.storeUserAttributes(directory.getId(), user.getName(), ImmutableMap.of(autoGroupsAddedForApplicationAttributeName(application), Collections.singleton(Boolean.TRUE.toString())));
        } catch (DirectoryPermissionException | DirectoryNotFoundException | UserNotFoundException e) {
            throw new OperationFailedException(e);
        }
    }

    private boolean userHasAttributeForApplicationAlreadySet(Application application, UserWithAttributes user) {
        return Boolean.parseBoolean(user.getValue(autoGroupsAddedForApplicationAttributeName(application)));
    }

    @VisibleForTesting
    static String autoGroupsAddedForApplicationAttributeName(Application application) {
        Preconditions.checkNotNull(application.getId());
        return DirectoryDefaultGroupMembershipResolver.AUTO_GROUPS_ADDED + ".app." + application.getId();
    }
}
