package com.atlassian.crowd.core.event;

import com.atlassian.crowd.event.TransactionAwareImmediateEvent;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.event.events.PluginEvent;
import com.atlassian.plugin.event.events.PluginFrameworkEvent;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Delays publishing any events if a transaction is active, until that transaction commits.
 * Events not published in transaction are unaffected. If the active transaction is rolled back, the events are not published.
 * <p>
 * Note that any transactional operations done in afterCommit(), need to be wrapped in a PROPAGATION_REQUIRES_NEW transaction,
 * (see {@link org.springframework.transaction.support.TransactionSynchronization#afterCommit()}). This is guaranteed by
 * {@link TransactionAwareEventDispatcher}
 * <p>
 * Since publishing an event during a transaction causes a {@link org.springframework.transaction.support.TransactionSynchronization}
 * to be registered for cases where many events are sent at once, it makes sense to use {@link #publishAll(Collection)}, to wrap them
 * in a single TransactionSynchronization, as transaction performance degrades notably with the number of distinct synchronizations registered.
 */
public class TransactionAwareEventPublisher extends DelegatingMultiEventPublisher implements EventPublisher, MultiEventPublisher {
    TransactionAwareEventPublisher(EventPublisher delegate) {
        super(delegate);
    }

    @Override
    public void publish(Object event) {
        if (shouldPostponeEvent(event)) {
            TransactionSynchronizationManager.registerSynchronization(createSynchronization(event));
        } else {
            delegate.publish(event);
        }
    }

    @Override
    public void publishAll(Collection<Object> events) {
        final Map<Boolean, List<Object>> partitionedEvents = events.stream().collect(Collectors.partitioningBy(this::shouldPostponeEvent));

        final List<Object> postponedEvents = partitionedEvents.get(true);
        final List<Object> immediateEvents = partitionedEvents.get(false);

        if (postponedEvents != null && !postponedEvents.isEmpty()) {
            TransactionSynchronizationManager.registerSynchronization(createSynchronization(postponedEvents));
        }

        if (immediateEvents != null) {
            immediateEvents.forEach(delegate::publish);
        }
    }

    private boolean shouldPostponeEvent(Object event) {
        // 'PluginFrameworkEvent' events need to be dispatched immediately because otherwise plugin system won't setup
        // 'PluginEvent' events need to be dispatched immediately e.g. in a case when UPM is being updated/uninstalled.
        //      For example when UPM is being updated PluginDisabledEvent events are being fired for all managed by UPM
        //      plugins and these events have to be handled before UPM is shutdown
        if (event instanceof PluginFrameworkEvent || event instanceof PluginEvent) {
            return false;
        } else {
            return !(event instanceof TransactionAwareImmediateEvent) &&
                    TransactionSynchronizationManager.isActualTransactionActive() &&
                    TransactionSynchronizationManager.isSynchronizationActive();
        }
    }

    private TransactionSynchronizationAdapter createSynchronization(final Object event) {
        return new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                delegate.publish(event);
            }
        };
    }

    private TransactionSynchronizationAdapter createSynchronization(final Collection<Object> events) {
        return new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                events.forEach(delegate::publish);
            }
        };
    }
}
