package com.atlassian.crowd.manager.recovery;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.embedded.api.PasswordCredential;
import com.atlassian.crowd.exception.ExpiredCredentialException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidAuthenticationException;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.InvalidGroupException;
import com.atlassian.crowd.exception.InvalidMembershipException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.MembershipAlreadyExistsException;
import com.atlassian.crowd.exception.MembershipNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.ReadOnlyGroupException;
import com.atlassian.crowd.exception.UserAlreadyExistsException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.avatar.AvatarReference;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupWithAttributes;
import com.atlassian.crowd.model.group.Membership;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.util.BoundedCount;

import com.google.common.collect.ImmutableMap;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.equalsInLowerCase;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Proxy directory that adds a temp admin account to Crowd and allow admin to login to recover from a broken deployment.
 * <p>
 * In case of a broken deployment or if the remote directory is down, the crowd admin will not be able to login to Crowd
 * if it's account info is stored in that remote directory.
 * <p>
 * In this case, sysops can enable this "Proxy" directory, which essentially adds a new directory contains an admin user
 * with a customisable password, and let the crowd admin to login to crowd and fix the deployment issue.
 */
public class RecoveryModeRemoteDirectory implements RemoteDirectory {
    private final long id;
    private final Map<String, String> attributes;
    private final String username;
    private final String password;

    public RecoveryModeRemoteDirectory(RecoveryModeDirectory directory) {
        checkNotNull(directory, "directory");
        this.id = checkNotNull(directory.getId(), "id");
        this.attributes = ImmutableMap.copyOf(directory.getAttributes());
        this.username = directory.getRecoveryUsername();
        this.password = directory.getRecoveryPassword();
    }

    @Override
    public long getDirectoryId() {
        return id;
    }

    @Override
    public void setDirectoryId(long directoryId) {
        throw new UnsupportedOperationException("Modifying ID is not supported");
    }

    @Override
    public String getDescriptiveName() {
        return "Recovery Mode Remote Directory";
    }

    @Override
    public void setAttributes(Map<String, String> attributes) {
        throw new UnsupportedOperationException("Modifying attributes is not supported");
    }

    @Override
    public User authenticate(String name, PasswordCredential credential) throws UserNotFoundException,
            InactiveAccountException, InvalidAuthenticationException, ExpiredCredentialException, OperationFailedException {
        if (credential.isEncryptedCredential()) {
            throw InvalidAuthenticationException.newInstanceWithName(name);
        }
        if (equalsInLowerCase(username, name)) {
            if (password.equals(credential.getCredential())) {
                return findUserWithAttributesByName(name);
            } else {
                throw new InvalidAuthenticationException("Invalid password credential");
            }
        } else {
            throw new UserNotFoundException(name);
        }
    }

    @Override
    public User findUserByName(String name) throws UserNotFoundException, OperationFailedException {
        return findUserWithAttributesByName(name);
    }

    @Override
    public UserWithAttributes findUserWithAttributesByName(String name) throws UserNotFoundException, OperationFailedException {
        if (equalsInLowerCase(username, name)) {
            return createRecoveryUser();
        } else {
            throw new UserNotFoundException(name);
        }
    }

    @Override
    public User findUserByExternalId(String externalId) throws UserNotFoundException, OperationFailedException {
        throw new OperationFailedException("Not supported");
    }

    @Override
    public User addUser(UserTemplate user, PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public UserWithAttributes addUser(UserTemplateWithAttributes user, PasswordCredential credential) throws InvalidUserException, InvalidCredentialException, UserAlreadyExistsException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public User updateUser(UserTemplate user) throws InvalidUserException, UserNotFoundException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void updateUserCredential(String username, PasswordCredential credential) throws UserNotFoundException, InvalidCredentialException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public User renameUser(String oldName, String newName) throws UserNotFoundException, InvalidUserException, UserAlreadyExistsException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void storeUserAttributes(String username, Map<String, Set<String>> attributes) throws UserNotFoundException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void removeUserAttributes(String username, String attributeName) throws UserNotFoundException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void removeUser(String name) throws UserNotFoundException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public <T> List<T> searchUsers(EntityQuery<T> query) throws OperationFailedException {
        // Technically this is not correct, we should attempt to check whether the query matches the recovery user
        // and return it if so.
        // However it does not seem to interfere with the recovery mode and it has the extra benefit of hiding the
        // recovery user from the UI. So for now this is implemented with a dummy version returning an empty list.
        return Collections.emptyList();
    }

    @Override
    public Group findGroupByName(String name) throws GroupNotFoundException, OperationFailedException {
        throw new GroupNotFoundException(name);
    }

    @Override
    public GroupWithAttributes findGroupWithAttributesByName(String name) throws GroupNotFoundException, OperationFailedException {
        throw new GroupNotFoundException(name);
    }

    @Override
    public Group addGroup(GroupTemplate group) throws InvalidGroupException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public Group updateGroup(GroupTemplate group) throws InvalidGroupException, GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public Group renameGroup(String oldName, String newName) throws GroupNotFoundException, InvalidGroupException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void storeGroupAttributes(String groupName, Map<String, Set<String>> attributes) throws GroupNotFoundException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void removeGroupAttributes(String groupName, String attributeName) throws GroupNotFoundException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void removeGroup(String name) throws GroupNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public <T> List<T> searchGroups(EntityQuery<T> query) throws OperationFailedException {
        return Collections.emptyList();
    }

    @Override
    public boolean isUserDirectGroupMember(String username, String groupName) throws OperationFailedException {
        return false;
    }

    @Override
    public boolean isGroupDirectGroupMember(String childGroup, String parentGroup) throws OperationFailedException {
        return false;
    }

    @Override
    public BoundedCount countDirectMembersOfGroup(String groupName, int querySizeHint) throws OperationFailedException {
        final MembershipQuery<String> membershipQuery = QueryBuilder
                .queryFor(String.class, EntityDescriptor.user())
                .childrenOf(EntityDescriptor.group())
                .withName(groupName)
                .startingAt(0)
                .returningAtMost(querySizeHint);

        return BoundedCount.fromCountedItemsAndLimit(searchGroupRelationships(membershipQuery).size(), querySizeHint);
    }

    @Override
    public void addUserToGroup(String username, String groupName) throws GroupNotFoundException, UserNotFoundException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void addGroupToGroup(String childGroup, String parentGroup) throws GroupNotFoundException, InvalidMembershipException, ReadOnlyGroupException, OperationFailedException, MembershipAlreadyExistsException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void removeUserFromGroup(String username, String groupName) throws GroupNotFoundException, UserNotFoundException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void removeGroupFromGroup(String childGroup, String parentGroup) throws GroupNotFoundException, InvalidMembershipException, MembershipNotFoundException, ReadOnlyGroupException, OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public void expireAllPasswords() throws OperationFailedException {
        throw new OperationFailedException("This is an immutable directory");
    }

    @Override
    public <T> List<T> searchGroupRelationships(MembershipQuery<T> query) throws OperationFailedException {
        return Collections.emptyList();
    }

    @Override
    public void testConnection() throws OperationFailedException {
    }

    @Override
    public boolean supportsInactiveAccounts() {
        return false;
    }

    @Override
    public boolean supportsNestedGroups() {
        return false;
    }

    /**
     * Does not support expiring passwords
     *
     * @return {@code false}, always
     */
    @Override
    public boolean supportsPasswordExpiration() {
        return false;
    }

    @Override
    public boolean supportsSettingEncryptedCredential() {
        return false;
    }

    @Override
    public boolean isRolesDisabled() {
        return true;
    }

    @Override
    public Iterable<Membership> getMemberships() throws OperationFailedException {
        return Collections.emptyList();
    }

    @Override
    public RemoteDirectory getAuthoritativeDirectory() {
        return this;
    }

    @Override
    public Set<String> getValues(String key) {
        return attributes.containsKey(key) ? Collections.singleton(attributes.get(key)) : null;
    }

    @Override
    public String getValue(String key) {
        return attributes.get(key);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    private UserWithAttributes createRecoveryUser() {
        UserTemplateWithAttributes recoveryUser = new UserTemplateWithAttributes(username, id);
        recoveryUser.setActive(true);
        recoveryUser.setDisplayName(SystemPropertyRecoveryModeService.RECOVERY_DISPLAY_NAME);
        recoveryUser.setEmailAddress(SystemPropertyRecoveryModeService.RECOVERY_EMAIL);
        return recoveryUser;
    }

    @Override
    public AvatarReference getUserAvatarByName(String username, int sizeHint) throws OperationFailedException {
        return null;
    }
}
