package com.atlassian.crowd.manager.recovery;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.DirectoryType;
import com.atlassian.crowd.embedded.api.OperationType;

import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A {@link Directory} implementation that holds a single in-memory user for recovery purposes.
 */
final class RecoveryModeDirectory implements Directory {
    private static final long serialVersionUID = 439749327493246L;

    // ID -2 ensures that there is no clash with 'real' persistent user directories maintained by Crowd
    // ID -1 is already used for application tokens, {@see com.atlassian.crowd.model.token.Token.APPLICATION_TOKEN_DIRECTORY_ID}
    static final long ID = -2L;

    private final String username;
    private final String password;

    private final long createdTimeMillis = System.currentTimeMillis();
    private final Map<String, String> attributes = Collections.emptyMap();

    RecoveryModeDirectory(String username, String password) {
        this.username = checkNotNull(username, "username");
        this.password = checkNotNull(password, "password");
    }

    @Override
    public Long getId() {
        return ID;
    }

    @Override
    public String getName() {
        return "Recovery Mode Directory";
    }

    @Override
    public boolean isActive() {
        return true;
    }

    @Override
    public String getEncryptionType() {
        return null;
    }

    @Override
    public Map<String, String> getAttributes() {
        return attributes;
    }

    @Override
    public Set<OperationType> getAllowedOperations() {
        return Collections.emptySet();
    }

    @Override
    public String getDescription() {
        return "Holds a single unmodifiable user that has system-wide permissions in the system. To use for recovering " +
                "from configuration issues that prevent any of the existing users from logging in.";
    }

    @Override
    public DirectoryType getType() {
        return DirectoryType.INTERNAL;
    }

    @Override
    public String getImplementationClass() {
        return getClass().getName();
    }

    @Override
    public Date getCreatedDate() {
        return new Date(createdTimeMillis);
    }

    @Override
    public Date getUpdatedDate() {
        return new Date(createdTimeMillis);
    }

    @Override
    public Set<String> getValues(String key) {
        return attributes.containsKey(key) ? Collections.singleton(attributes.get(key)) : null;
    }

    @Override
    public String getValue(String key) {
        return attributes.get(key);
    }

    @Override
    public Set<String> getKeys() {
        return attributes.keySet();
    }

    @Override
    public boolean isEmpty() {
        return attributes.isEmpty();
    }

    public String getRecoveryUsername() {
        return username;
    }

    public String getRecoveryPassword() {
        return password;
    }
}
