package com.atlassian.crowd.manager.recovery;

import java.util.Collection;
import java.util.List;

import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.ApplicationFactory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.event.EventStore;
import com.atlassian.crowd.manager.application.ApplicationServiceGeneric;
import com.atlassian.crowd.manager.application.SearchStrategyFactory;
import com.atlassian.crowd.manager.avatar.AvatarProvider;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.manager.permission.PermissionManager;
import com.atlassian.crowd.manager.webhook.WebhookRegistry;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.event.api.EventPublisher;

import com.google.common.collect.ImmutableList;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An extension of {@link ApplicationServiceGeneric} that supports recovery mode. When recovery mode is activated,
 * an extra directory as provided by the {@link RecoveryModeService#getRecoveryDirectory() RecoveryModeService} will
 * be prepended to the list of directories for any application.
 *
 * @since 2.7.2
 */
public class RecoveryModeAwareApplicationService extends ApplicationServiceGeneric {
    private final RecoveryModeService recoveryModeService;
    private final ApplicationFactory applicationFactory;

    public RecoveryModeAwareApplicationService(DirectoryManager directoryManager,
                                               SearchStrategyFactory searchStrategyFactory,
                                               PermissionManager permissionManager,
                                               EventPublisher eventPublisher,
                                               EventStore eventStore,
                                               WebhookRegistry webhookRegistry,
                                               AvatarProvider avatarProvider,
                                               ApplicationFactory applicationFactory,
                                               RecoveryModeService recoveryModeService) {
        super(directoryManager, searchStrategyFactory, permissionManager, eventPublisher, eventStore, webhookRegistry, avatarProvider);
        this.applicationFactory = checkNotNull(applicationFactory);
        this.recoveryModeService = checkNotNull(recoveryModeService, "recoveryModeService");
    }

    @Override
    protected List<Directory> getActiveDirectories(Application application) {
        List<Directory> directories = super.getActiveDirectories(application);
        if (recoveryModeService.isRecoveryModeOn() && application.equals(applicationFactory.getApplication())) {
            // add extra 'recovery mode' directory
            return ImmutableList.<Directory>builder()
                    .add(recoveryModeService.getRecoveryDirectory())
                    .addAll(directories)
                    .build();
        } else {
            return directories;
        }
    }

    @Override
    public boolean isUserAuthorised(Application application, String username) {
        if (recoveryModeService.isRecoveryModeOn() &&
                IdentifierUtils.equalsInLowerCase(username, recoveryModeService.getRecoveryUsername()) &&
                application.equals(applicationFactory.getApplication())) {
            return true; // the recovery user is permitted to attempt authentication in Crowd
        } else {
            return super.isUserAuthorised(application, username);
        }
    }
}
