package com.atlassian.crowd.manager.directory.monitor.poller;

import java.io.Serializable;
import java.util.Date;

import com.atlassian.crowd.directory.monitor.poller.DirectoryPoller;
import com.atlassian.crowd.manager.directory.SynchronisationMode;
import com.atlassian.crowd.manager.directory.monitor.DirectoryMonitorRegistrationException;
import com.atlassian.crowd.manager.directory.monitor.DirectoryMonitorUnregistrationException;
import com.atlassian.scheduler.SchedulerService;
import com.atlassian.scheduler.SchedulerServiceException;
import com.atlassian.scheduler.config.JobConfig;
import com.atlassian.scheduler.config.JobId;
import com.atlassian.scheduler.config.Schedule;
import com.atlassian.scheduler.status.JobDetails;

import com.google.common.collect.ImmutableMap;

import static com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerJobRunner.JOB_RUNNER_KEY;
import static com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerJobRunner.PARAM_DIRECTORY_ID;
import static com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerJobRunner.PARAM_SYNC_MODE;
import static com.atlassian.scheduler.config.RunMode.RUN_ONCE_PER_CLUSTER;

/**
 * Implementation of DirectoryPollerManager that uses Atlassian Scheduler
 *
 * @since 2.8
 */
public class AtlassianSchedulerDirectoryPollerManager implements DirectoryPollerManager {

    private final SchedulerService schedulerService;

    public AtlassianSchedulerDirectoryPollerManager(SchedulerService schedulerService) {
        this.schedulerService = schedulerService;
    }

    @Override
    public void addPoller(DirectoryPoller poller) throws DirectoryMonitorRegistrationException {
        Date startDelay = new Date(getStartDelay());
        JobConfig config = JobConfig.forJobRunnerKey(JOB_RUNNER_KEY)
                .withRunMode(RUN_ONCE_PER_CLUSTER)
                .withSchedule(Schedule.forInterval(poller.getPollingInterval() * 1000, startDelay))
                .withParameters(ImmutableMap.<String, Serializable>builder()
                        .put(PARAM_DIRECTORY_ID, poller.getDirectoryID())
                        .build());

        try {
            schedulerService.scheduleJob(getJobId(poller), config);
        } catch (SchedulerServiceException e) {
            throw new DirectoryMonitorRegistrationException(e);
        }
    }

    @Override
    public boolean hasPoller(long directoryID) {
        return schedulerService.getJobDetails(getJobId(directoryID)) != null;
    }

    @Override
    public void triggerPoll(long directoryID, SynchronisationMode syncMode) {
        JobConfig config = JobConfig.forJobRunnerKey(JOB_RUNNER_KEY)
                .withRunMode(RUN_ONCE_PER_CLUSTER)
                .withSchedule(Schedule.runOnce(null))
                .withParameters(ImmutableMap.<String, Serializable>builder()
                        .put(PARAM_DIRECTORY_ID, directoryID)
                        .put(PARAM_SYNC_MODE, syncMode)
                        .build());
        try {
            // Schedule with generated ID to not overwrite the run config for the automatic sync schedule.
            schedulerService.scheduleJobWithGeneratedId(config);
        } catch (SchedulerServiceException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public boolean removePoller(long directoryID) throws DirectoryMonitorUnregistrationException {
        boolean exists = hasPoller(directoryID);
        schedulerService.unscheduleJob(getJobId(directoryID));
        return exists;
    }

    @Override
    public void removeAllPollers() {
        for (JobDetails job : schedulerService.getJobsByJobRunnerKey(JOB_RUNNER_KEY)) {
            schedulerService.unscheduleJob(job.getJobId());
        }
    }

    protected long getStartDelay() {
        return System.currentTimeMillis() + Long.getLong("crowd.polling.startdelay", 5000L);
    }

    private JobId getJobId(DirectoryPoller poller) {
        return getJobId(poller.getDirectoryID());
    }

    private JobId getJobId(long directoryId) {
        return JobId.of(DirectoryPollerManager.class.getName() + "." + directoryId);
    }

}
