package com.atlassian.crowd.manager.application;

import java.util.List;

import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

/**
 * A set of querying across a collection of active directories assigned to an application.
 * <p>
 * A strategy is resolved using a {@link SearchStrategyFactory} and can be optimised for specific configurations of
 * directory.
 *
 * @see SearchStrategyFactory
 * @since 2.9
 */
public interface SearchStrategy {
    /**
     * Returns a {@code List<User>} or {@code List<String>} matching the search criteria defined in the query.
     * <p>
     * The users will be returned in a stable order including across pagination boundaries (excluding modification).
     *
     * @param query the search query.
     * @return {@code List<User>} user objects or {@code List<String>} usernames, depending on the query.
     */
    <T> List<T> searchUsers(EntityQuery<T> query);

    /**
     * Returns a {@code List<Group>} matching the search criteria defined in the query.
     * <p>
     * The groups will be returned in a stable order including across pagination boundaries (excluding modification).
     *
     * @param query the search query.
     * @return {@code List<Group>} group objects or {@code List<String>} groupnames, depending on the query.
     */
    <T> List<T> searchGroups(EntityQuery<T> query);

    /**
     * Searches for direct group relationships in any of the application's active assigned directories.
     * <p>
     * When searching for the groups an entity is a member of,
     * the {@link com.atlassian.crowd.model.application.Application#isMembershipAggregationEnabled() membership aggregation semantic}
     * will determine whether only the owning directory (when {@code false}) or all directories (when {@code true}) will be searched.
     *
     * @param query membership query.
     * @return List of {@link com.atlassian.crowd.model.user.User} entities,
     * {@link com.atlassian.crowd.model.group.Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria.
     */
    <T> List<T> searchDirectGroupRelationships(MembershipQuery<T> query);

    /**
     * Searches for direct and indirect (nested) group relationships in any of the application's active assigned directories.
     * <p>
     * If the directory does not support nested groups, this call will be equivalent to {@link com.atlassian.crowd.manager.directory.DirectoryManager#searchDirectGroupRelationships(long, com.atlassian.crowd.search.query.membership.MembershipQuery)}.
     * <p>
     * <b>WARNING: this method could be very slow if the underlying RemoteDirectory does not employ caching.</b>
     * <p>
     * When searching for the groups a user is a member of only the directory of the user (as determined by findUserByName)
     * is searched.  When searching for memberships of a group or groups a group is a member of all directories are searched and the results amalgamated.
     *
     * @param query membership query.
     * @return List of {@link com.atlassian.crowd.model.user.User} entities,
     * {@link com.atlassian.crowd.model.group.Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria.
     */
    <T> List<T> searchNestedGroupRelationships(MembershipQuery<T> query);
}
