package com.atlassian.crowd.manager.application;

import java.util.ArrayList;
import java.util.ConcurrentModificationException;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.api.SearchRestriction;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.Combine;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.builder.Restriction;
import com.atlassian.crowd.search.query.QueryUtils;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.entity.restriction.NullRestrictionImpl;
import com.atlassian.crowd.search.query.entity.restriction.constants.UserTermKeys;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.crowd.model.DirectoryEntities.LOWER_NAME_FUNCTION;
import static com.atlassian.crowd.model.DirectoryEntities.NAME_FUNCTION;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A in-memory {@link SearchStrategy} which only return memberships associated with the canonical users directory.
 * <p>
 * This is considered the worse case {@link SearchStrategy} to use as searching across multiple directories
 * will be done in-memory, potentially consuming a lot of memory. This is the same (in spirit) as what Crowd 2.8
 * and earlier would do by default.
 *
 * @see com.atlassian.crowd.model.application.Application#isMembershipAggregationEnabled()
 * @since 2.9
 */
public class InMemoryNonAggregatingSearchStrategy extends AbstractInMemorySearchStrategy {
    private static final Logger logger = LoggerFactory.getLogger(InMemoryNonAggregatingSearchStrategy.class);

    public InMemoryNonAggregatingSearchStrategy(DirectoryManager directoryManager, List<Directory> activeDirectories) {
        super(directoryManager, activeDirectories);
    }

    @Override
    public <T> List<T> searchDirectGroupRelationships(final MembershipQuery<T> query) {
        // restrict searches to these three types. Instances of com.atlassian.crowd.embedded.api.user.{User|Group}
        // are not accepted
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class, User.class);

        // even when not aggregating, we still want to sort and constrain the results
        // Excluding duplicates naturally disallows the inclusion of shadowing entities, given that we aggregate in
        // directory order.
        final ResultsAggregator<T> results = ResultsAggregator.with(getAggregatingAndSortingComparatorFor(query.getReturnType()), query);
        final Map<Long, List<DirectoryEntity>> allEntities = new LinkedHashMap<>();
        if (query.isFindChildren()) {
            // find all non-shadowed children
            for (final Directory directory : activeDirectories) {
                final List<DirectoryEntity> itemsToAdd;
                // Unfortunate repetition due to a lack of real algebraic datatypes in Java
                if (query.getReturnType() == User.class || query.getReturnType() == Group.class) {
                    // Casting is required here because the underlying directory will need to access the exact type
                    // passed to the underlying query at runtime, so we can't create a new query with a more general
                    // type.
                    final MembershipQuery<DirectoryEntity> castQuery = (MembershipQuery<DirectoryEntity>) query;
                    itemsToAdd = doDirectDirectoryMembershipQuery(castQuery, directory.getId());
                } else if (query.getReturnType() == String.class && query.getEntityToReturn().equals(EntityDescriptor.user())) {
                    itemsToAdd = usersInDirectory(directory.getId(), (List<String>) doDirectDirectoryMembershipQuery(query, directory.getId()));
                } else if (query.getReturnType() == String.class && query.getEntityToReturn().equals(EntityDescriptor.group())) {
                    itemsToAdd = groupsInDirectory(directory.getId(), (List<String>) doDirectDirectoryMembershipQuery(query, directory.getId()));
                } else {
                    throw new IllegalArgumentException("Unsupported query return type or entity: " + query.toString());
                }
                allEntities.put(directory.getId(), new ArrayList<>(itemsToAdd));
            }
            removeNonCanonicalEntities(allEntities, activeDirectories, query);
            addResultsFromDirectories(query, results, allEntities);
        } else {
            // only search the directory of the child entity
            try {
                final DirectoryEntity entity = findEntityByName(query.getEntityNameToMatch(), query.getEntityToMatch());
                results.addAll(doDirectDirectoryMembershipQuery(query, entity.getDirectoryId()));
            } catch (ObjectNotFoundException e) {
                // Don't worry.  No results.
            }
        }

        return results.constrainResults();
    }

    private <T> void addResultsFromDirectories(final MembershipQuery<T> query, final ResultsAggregator<T> results, final Map<Long, List<DirectoryEntity>> allEntities) {
        for (final List<DirectoryEntity> directoryEntities : allEntities.values()) {
            if (query.getReturnType() == User.class || query.getReturnType() == Group.class) {
                results.addAll((Iterable<? extends T>) directoryEntities);
            } else if (query.getReturnType() == String.class
                    && (query.getEntityToReturn().equals(EntityDescriptor.user())
                    || query.getEntityToReturn().equals(EntityDescriptor.group()))) {
                results.addAll((Iterable<T>) Iterables.transform(directoryEntities, NAME_FUNCTION));
            }
        }
    }

    @Override
    public <T> List<T> searchNestedGroupRelationships(final MembershipQuery<T> query) {
        // restrict searches to these three types. Instances of com.atlassian.crowd.embedded.api.user.{User|Group}
        // are not accepted
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class, User.class);

        // even when not aggregating, we still want to sort and constraint the results
        final ResultsAggregator<T> results = ResultsAggregator.with(getAggregatingAndSortingComparatorFor(query.getReturnType()), query);
        final Map<Long, List<DirectoryEntity>> allEntities = new LinkedHashMap<>();
        if (query.isFindChildren()) {
            // find all non-shadowed children
            for (final Directory directory : activeDirectories) {
                final List<DirectoryEntity> itemsToAdd;
                // Unfortunate repetition due to a lack of real algebraic datatypes in Java
                if (query.getReturnType() == User.class || query.getReturnType() == Group.class) {
                    // Casting is required here because the underlying directory will need to access the exact type
                    // passed to the underlying query at runtime, so we can't create a new query with a more general
                    // type.
                    final MembershipQuery<DirectoryEntity> castQuery = (MembershipQuery<DirectoryEntity>) query;
                    itemsToAdd = doNestedDirectoryMembershipQuery(castQuery, directory.getId());
                } else if (query.getReturnType() == String.class && query.getEntityToReturn().equals(EntityDescriptor.user())) {
                    final MembershipQuery<DirectoryEntity> expandedQuery = (MembershipQuery<DirectoryEntity>) new MembershipQuery(query, User.class);
                    itemsToAdd = doNestedDirectoryMembershipQuery(expandedQuery, directory.getId());
                } else if (query.getReturnType() == String.class && query.getEntityToReturn().equals(EntityDescriptor.group())) {
                    final MembershipQuery<DirectoryEntity> expandedQuery = (MembershipQuery<DirectoryEntity>) new MembershipQuery(query, Group.class);
                    itemsToAdd = doNestedDirectoryMembershipQuery(expandedQuery, directory.getId());
                } else {
                    throw new IllegalArgumentException("Unsupported query return type or entity: " + query.toString());
                }
                allEntities.put(directory.getId(), new ArrayList<>(itemsToAdd));
            }
            removeNonCanonicalEntities(allEntities, activeDirectories, query);
            addResultsFromDirectories(query, results, allEntities);
        } else {
            // only search the directory of the child entity
            try {
                final DirectoryEntity entity = findEntityByName(query.getEntityNameToMatch(), query.getEntityToMatch());
                results.addAll(doNestedDirectoryMembershipQuery(query, entity.getDirectoryId()));
            } catch (ObjectNotFoundException e) {
                // Don't worry.  No results.
            }
        }

        return results.constrainResults();
    }

    /**
     * <p>
     * Given a username is duplicated in several {@link Directory user directories} under the same
     * {@link Application application}.
     * </p>
     * <p>The user in the first directory, according to directory ordering is considered to
     * be the canonical user for the given username, and the other users are shadowed and thus not to be returned from
     * searches.</p>
     *
     * <p>
     * This method removes all entities which were found for a given directory, but were not canonical in it (they existed
     * in a preceding directory, not necessarily in the allEntities map). The allEntities map will be mutated in the
     * process. For example:
     *
     * Given the following directories and entities:
     *
     * <ul>
     * <li>Directory 1 - Users A, B, C</li>
     * <li>Directory 2 - Users C, D, E</li>
     * </ul>
     *
     * At this point users A, B and C are known as canonical in the Directory 1 and will be removed from the
     * succeeding directories. This will leave users D and E in Directory 2. However their canonicality in Directory 2
     * is uncertain as they may exist in Directory 1, but may have not been matched by the query that was used to
     * retrieve the users. Because of this Directory 1 will be queried for users D and E and the results that were
     * found will be removed from all succeeding directories. If user D is found in Directory 1 this will give us the
     * following result:
     *
     * <ul>
     * <li>Directory 1 - Users A, B, C</li>
     * <li>Directory 2 - User E</li>
     * </ul>
     *
     * User D was not included in the result set due to being canonical in Directory 1, but not matching the query
     * used to retrieve the entities.
     * </p>
     *
     * @param allEntities       a directoryId to list of entities map containing the entities to check for canonicality.
     *                          The map must have a predictable iteration order
     * @param activeDirectories the directories to lookup, should match the keyset of the map
     * @param originalQuery     the query used to look up the entities that will be determined as canonical
     */
    private <T> void removeNonCanonicalEntities(final Map<Long, List<DirectoryEntity>> allEntities, final List<Directory> activeDirectories, final MembershipQuery<T> originalQuery) {
        removeEntitiesFoundInTheFirstDirectory(allEntities, activeDirectories);
        for (int i = 1; i < activeDirectories.size(); i++) {
            final Directory directory = activeDirectories.get(i);
            final List<Directory> succeedingDirectories;
            succeedingDirectories = activeDirectories.stream().skip(i + 1).collect(Collectors.toList());
            final List<DirectoryEntity> directoryEntities = allEntities.get(directory.getId());
            if (directoryEntities.size() == 0) {
                continue;
            }
            final List<Directory> precedingDirectories = activeDirectories.subList(0, i);
            for (final Directory precedingDirectory : precedingDirectories) {
                final List<String> matchingNamesInDirectory = findRemainingEntities(originalQuery, directoryEntities, precedingDirectory);
                final Set<String> canonicalEntityNames = new HashSet<>((Lists.transform(matchingNamesInDirectory, IdentifierUtils.TO_LOWER_CASE)));
                removeCanonicalUsers(canonicalEntityNames, allEntities.get(directory.getId()));
                succeedingDirectories.stream().map(dir -> allEntities.get(dir.getId())).forEach(users -> removeCanonicalUsers(canonicalEntityNames, users));
            }

        }
    }

    private void removeEntitiesFoundInTheFirstDirectory(final Map<Long, List<DirectoryEntity>> allEntities, final List<Directory> activeDirectories) {
        final Directory directory = activeDirectories.get(0);
        final List<DirectoryEntity> directoryEntities = allEntities.get(directory.getId());
        final HashSet<String> canonicalEntityNames = new HashSet<>(Lists.transform(directoryEntities, LOWER_NAME_FUNCTION));
        final List<Directory> succeedingDirectories = activeDirectories.stream().skip(1).collect(Collectors.toList());
        succeedingDirectories.stream().map(dir -> allEntities.get(dir.getId())).forEach(users -> removeCanonicalUsers(canonicalEntityNames, users));
    }

    private <T> List<String> findRemainingEntities(final MembershipQuery<T> originalQuery, final List<DirectoryEntity> directoryEntities, final Directory precedingDirectory) {
        final EntityQuery<String> remainingEntitiesQuery = createRemainingEntitiesQuery(originalQuery, directoryEntities);
        final SingleDirectorySearchStrategy searchStrategy = new SingleDirectorySearchStrategy(directoryManager, precedingDirectory.getId());
        final List<String> matchingNamesInDirectory;
        if (remainingEntitiesQuery.getEntityDescriptor().equals(EntityDescriptor.user())) {
            matchingNamesInDirectory = searchStrategy.searchUsers(remainingEntitiesQuery);
        } else if (remainingEntitiesQuery.getEntityDescriptor().equals(EntityDescriptor.group())) {
            matchingNamesInDirectory = searchStrategy.searchGroups(remainingEntitiesQuery);
        } else {
            throw new IllegalArgumentException("Type " + remainingEntitiesQuery.getReturnType() + "is not supported");
        }
        return matchingNamesInDirectory;
    }

    /**
     * This will create a query for the specified directory entities.
     *
     * @param originalQuery
     * @param directoryEntities
     * @param <T>
     * @return
     */
    private <T> EntityQuery<String> createRemainingEntitiesQuery(final MembershipQuery<T> originalQuery, final List<DirectoryEntity> directoryEntities) {
        final List<SearchRestriction> remainingEntitiesRestrictions = directoryEntities.stream().map(user -> Restriction.on(UserTermKeys.USERNAME).exactlyMatching(user.getName())).collect(Collectors.toList());
        final SearchRestriction queryRestriction = remainingEntitiesRestrictions.size() > 0 ? Combine.anyOf(remainingEntitiesRestrictions) : NullRestrictionImpl.INSTANCE;
        return QueryBuilder.queryFor(String.class, originalQuery.getEntityToReturn(),
                queryRestriction, 0, EntityQuery.ALL_RESULTS);
    }

    private void removeCanonicalUsers(final Set<String> canonicalUserNames, final List<DirectoryEntity> usersInSucceedingDirectories) {
        Iterables.removeIf(usersInSucceedingDirectories, entity -> canonicalUserNames.contains(LOWER_NAME_FUNCTION.apply(entity)));
    }

    private User findUserByName(final String name) throws UserNotFoundException {
        for (final Directory directory : activeDirectories) {
            try {
                final User user = directoryManager.findUserByName(directory.getId(), name);
                logger.debug("Located user '{}' in directory {} '{}'", user.getName(), directory.getId(), directory.getName());
                return user;
            } catch (final UserNotFoundException e) {
                // user not in directory, keep cycling
            } catch (final DirectoryNotFoundException e) {
                logger.debug("Directory {} '{}' was active at the start of the loop, but can no longer be found", directory.getId(), directory.getName());
                throw concurrentModificationExceptionForDirectoryIteration(e);
            } catch (final OperationFailedException e) {
                // directory has some massive error, keep cycling
                logger.error(e.getMessage(), e);
            }
        }

        // could not find user in any of the directories
        throw new UserNotFoundException(name);
    }

    private Group findGroupByName(final String name) throws GroupNotFoundException {
        for (final Directory directory : activeDirectories) {
            try {
                return directoryManager.findGroupByName(directory.getId(), name);
            } catch (final GroupNotFoundException e) {
                // group not in directory, keep cycling
            } catch (final DirectoryNotFoundException e) {
                throw concurrentModificationExceptionForDirectoryIteration(e);
            } catch (final OperationFailedException e) {
                // directory has some massive error, keep cycling
                logger.error(e.getMessage(), e);
            }
        }

        // could not find group in any of the directories
        throw new GroupNotFoundException(name);
    }

    private DirectoryEntity findEntityByName(String entityName, EntityDescriptor entityDescriptor)
            throws ObjectNotFoundException {
        if (entityDescriptor == EntityDescriptor.user()) {
            return findUserByName(entityName);
        } else if (entityDescriptor == EntityDescriptor.group()) {
            return findGroupByName(entityName);
        } else {
            throw new IllegalArgumentException("Expected a User or Group");
        }
    }


    private static List<DirectoryEntity> usersInDirectory(final long directoryId, List<String> names) {
        return Lists.transform(names, new Function<String, DirectoryEntity>() {
            @Override
            public DirectoryEntity apply(String name) {
                return new LightweightUser(directoryId, name);
            }
        });
    }

    private static List<DirectoryEntity> groupsInDirectory(final long directoryId, List<String> names) {
        return Lists.transform(names, new Function<String, DirectoryEntity>() {
            @Override
            public DirectoryEntity apply(String name) {
                return new LightweightGroup(directoryId, name);
            }
        });
    }

    private static ConcurrentModificationException concurrentModificationExceptionForDirectoryIteration(
            DirectoryNotFoundException e) {
        final ConcurrentModificationException concurrentModificationException = new ConcurrentModificationException(
                "Directory mapping was removed while iterating through directories");
        concurrentModificationException.initCause(e);
        return concurrentModificationException;
    }

    /**
     * isCanonical needs to distinguish between the types of entities, but doesn't need any more information.
     * Introduce this class to allow a name to be tagged with an entity type.
     */
    private static abstract class LightweightDirectoryEntity implements DirectoryEntity {
        private final long directoryId;
        private final String name;

        LightweightDirectoryEntity(long directoryId, String name) {
            this.directoryId = directoryId;
            this.name = checkNotNull(name);
        }

        @Override
        public long getDirectoryId() {
            return directoryId;
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public int hashCode() {
            return 31 * (31 + (int) (directoryId ^ (directoryId >>> 32))) + IdentifierUtils.toLowerCase(name).hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (obj == null || getClass() != obj.getClass()) {
                return false;
            }

            LightweightDirectoryEntity other = (LightweightDirectoryEntity) obj;

            return directoryId == other.directoryId && IdentifierUtils.equalsInLowerCase(name, other.name);
        }
    }

    private static class LightweightUser extends LightweightDirectoryEntity {
        LightweightUser(long directoryId, String name) {
            super(directoryId, name);
        }
    }

    private static class LightweightGroup extends LightweightDirectoryEntity {
        LightweightGroup(long directoryId, String name) {
            super(directoryId, name);
        }
    }
}
