package com.atlassian.crowd.manager.application;

import java.util.Collection;
import java.util.List;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.query.QueryUtils;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

/**
 * An in-memory {@link SearchStrategy} which aggregates memberships across multiple directories.
 * <p>
 * This is considered the worse case {@link SearchStrategy} to use as searching across multiple directories
 * will be done in-memory, potentially consuming a lot of memory. This is the same (in spirit) as what Crowd 2.8
 * and earlier would do by default.
 *
 * @see com.atlassian.crowd.model.application.Application#isMembershipAggregationEnabled()
 * @since 2.9
 */
public class InMemoryAggregatingSearchStrategy extends AbstractInMemorySearchStrategy {
    public InMemoryAggregatingSearchStrategy(DirectoryManager directoryManager, List<Directory> activeDirectories) {
        super(directoryManager, activeDirectories);
    }

    @Override
    public <T> List<T> searchDirectGroupRelationships(MembershipQuery<T> query) {
        // restrict searches to these three types. Instances of com.atlassian.crowd.embedded.api.user.{User|Group}
        // are not accepted
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class, User.class);

        // even when not aggregating, we still want to sort and constrain the results
        // Excluding duplicates naturally disallows the inclusion of shadowing entities, given that we aggregate in
        // directory order.
        ResultsAggregator<T> results = ResultsAggregator.with(getAggregatingAndSortingComparatorFor(query.getReturnType()), query);

        for (final Directory directory : activeDirectories) {
            results.addAll(doDirectDirectoryMembershipQuery(query, directory.getId()));
        }

        return results.constrainResults();
    }

    @Override
    public <T> List<T> searchNestedGroupRelationships(MembershipQuery<T> query) {
        // restrict searches to these three types. Instances of com.atlassian.crowd.embedded.api.user.{User|Group}
        // are not accepted
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class, User.class);

        // even when not aggregating, we still want to sort and constraint the results
        ResultsAggregator<T> results = ResultsAggregator.with(getAggregatingAndSortingComparatorFor(query.getReturnType()), query);

        for (final Directory directory : activeDirectories) {
            results.addAll(doNestedDirectoryMembershipQuery(query, directory.getId()));
        }

        return results.constrainResults();
    }
}
