package com.atlassian.crowd.manager.application;

import java.util.List;
import java.util.Optional;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.application.Application;

import com.google.common.collect.Iterables;

import static java.util.Objects.requireNonNull;

/**
 * A {@link SearchStrategyFactory} which will delegate to the {@link DirectoryManager} for all of its
 * {@link SearchStrategy search strategies}.
 * <p>
 * This factory is guaranteed to return non-empty for any application
 * and configuration of directories
 *
 * @see NoDirectorySearchStrategy
 * @see SingleDirectorySearchStrategy
 * @see InMemoryAggregatingSearchStrategy
 * @since 2.9
 */
public class DefaultSearchStrategyFactory implements SearchStrategyFactory {
    private final DirectoryManager directoryManager;

    public DefaultSearchStrategyFactory(DirectoryManager directoryManager) {
        this.directoryManager = requireNonNull(directoryManager, "directoryManager");
    }

    @Override
    public Optional<SearchStrategy> createSearchStrategy(Application application, List<Directory> activeDirectories) {
        SearchStrategy strategy;
        if (activeDirectories.isEmpty()) {
            strategy = NoDirectorySearchStrategy.INSTANCE;
        } else if (activeDirectories.size() == 1) {
            strategy = new SingleDirectorySearchStrategy(directoryManager, Iterables.getOnlyElement(activeDirectories).getId());
        } else if (application.isMembershipAggregationEnabled()) {
            strategy = new InMemoryAggregatingSearchStrategy(directoryManager, activeDirectories);
        } else {
            strategy = new InMemoryNonAggregatingSearchStrategy(directoryManager, activeDirectories);
        }
        return Optional.of(strategy);
    }
}
