package com.atlassian.crowd.manager.application;

import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.List;

import javax.annotation.Nullable;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.ObjectNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.NameComparator;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.QueryUtils;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.crowd.model.DirectoryEntities.NAME_FUNCTION;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * An abstract {@link SearchStrategy} which searches across multiple directories in memory for users and groups
 * <p>
 * This is considered the worse case {@link SearchStrategy} to use as aggregation across multiple directories
 * will be done in-memory, potentially consuming a lot of memory. This is the same (in spirit) as what Crowd 2.8
 * and earlier would do by default.
 *
 * @since 2.9
 */
public abstract class AbstractInMemorySearchStrategy implements SearchStrategy {
    protected final DirectoryManager directoryManager;
    protected final List<Directory> activeDirectories;

    public AbstractInMemorySearchStrategy(DirectoryManager directoryManager, List<Directory> activeDirectories) {
        this.directoryManager = directoryManager;
        this.activeDirectories = activeDirectories;
    }

    @Override
    public <T> List<T> searchUsers(EntityQuery<T> query) {
        return searchUsers(query, getAggregatingAndSortingComparatorFor(query.getReturnType()));
    }

    private <T, K extends Comparable<? super K>> List<T> searchUsers(final EntityQuery<T> query, final Function<? super T, K> comparator) {
        // explicitly disallow searches for embedded users
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, User.class);

        ResultsAggregator<T> results = ResultsAggregator.with(comparator, query);

        // TODO: implement this so it's more efficient with indexing + max results
        final EntityQuery<T> totalQuery = QueryBuilder.queryFor(query.getReturnType(), query.getEntityDescriptor(),
                query.getSearchRestriction(), 0, EntityQuery.ALL_RESULTS);

        for (final Directory directory : activeDirectories) {
            // perform the search
            List<T> users = new SingleDirectorySearchStrategy(directoryManager, directory.getId())
                    .searchUsers(totalQuery);
            results.addAll(users);
        }

        return results.constrainResults();
    }


    @Override
    public <T> List<T> searchGroups(EntityQuery<T> query) {
        // explicitly disallow searches for embedded groups
        QueryUtils.checkAssignableFrom(query.getReturnType(), String.class, Group.class);

        ResultsAggregator<T> results = ResultsAggregator.with(getAggregatingAndSortingComparatorFor(query.getReturnType()), query);

        // TODO: implement this so it's more efficient with indexing + max results
        final EntityQuery<T> totalQuery = QueryBuilder.queryFor(query.getReturnType(), query.getEntityDescriptor(),
                query.getSearchRestriction(), 0, EntityQuery.ALL_RESULTS);

        for (final Directory directory : activeDirectories) {
            // perform the search
            final List<T> groups = new SingleDirectorySearchStrategy(directoryManager, directory.getId())
                    .searchGroups(totalQuery);
            results.addAll(groups);
        }

        return results.constrainResults();
    }

    /**
     * Searches for direct group relationships in a single directory.
     *
     * @param query       membership query.
     * @param directoryId Directory to search.
     * @return List of {@link User} entities,
     * {@link Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria.
     */
    protected <T> List<T> doDirectDirectoryMembershipQuery(final MembershipQuery<T> query, final long directoryId) {
        final MembershipQuery<T> totalQuery = QueryBuilder.createMembershipQuery(EntityQuery.ALL_RESULTS, 0, query.isFindChildren(),
                query.getEntityToReturn(), query.getReturnType(), query.getEntityToMatch(), query.getEntityNameToMatch());

        return new SingleDirectorySearchStrategy(directoryManager, directoryId)
                .searchDirectGroupRelationships(totalQuery);
    }

    /**
     * Searches for direct and indirect (nested) group relationships in a single directory.
     *
     * @param query       membership query.
     * @param directoryId Directory to search.
     * @return List of {@link User} entities,
     * {@link Group} entities,
     * {@link String} usernames or {@link String} group names matching the query criteria.
     */
    protected <T> List<T> doNestedDirectoryMembershipQuery(final MembershipQuery<T> query, final long directoryId) {
        final MembershipQuery<T> totalQuery = QueryBuilder.createMembershipQuery(EntityQuery.ALL_RESULTS, 0, query.isFindChildren(),
                query.getEntityToReturn(), query.getReturnType(), query.getEntityToMatch(), query.getEntityNameToMatch());

        return new SingleDirectorySearchStrategy(directoryManager, directoryId)
                .searchNestedGroupRelationships(totalQuery);
    }

    /**
     * Returns a comparator for aggregating and sorting the results.
     * <p>
     * For:
     * <ul>
     * <li>String: names are aggregated based on case-insensitive comparison.
     * <li>User: users are aggregated based on case-insensitive comparison and directoryId. Two users with the same
     * name but from different directories are treated as different users.
     * <li>Group: groups are aggregated based on case-insensitive comparison. Two groups with the same name but
     * from different directories are treated as the same group.
     * </ul>
     *
     * @param type type we are aggregating.
     * @return comparator for the type.
     */
    protected static <T> Function<T, String> getAggregatingAndSortingComparatorFor(final Class<T> type) {
        if (String.class.isAssignableFrom(type)) {
            return NameComparator.normaliserOf(type);
        } else if (User.class.isAssignableFrom(type)) {
            // by default user names should always be unique. only {@link #searchUsersAllowingDuplicateNames} will allow
            // users with the same user name but different directory ID to be returned.
            return NameComparator.normaliserOf(type);
        } else if (Group.class.isAssignableFrom(type)) {
            return NameComparator.normaliserOf(type);
        } else {
            throw new IllegalArgumentException("Cannot find normaliser for type: " + type.getCanonicalName());
        }
    }
}
