package com.atlassian.crowd.directory.loader;

import com.atlassian.crowd.directory.DirectoryProperties;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.directory.ldap.cache.DirectoryCacheFactory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.crowd.manager.directory.monitor.DirectoryMonitorManager;

/**
 * A subclass of {@link DbCachingRemoteDirectoryInstanceLoader} which allows DB caching to be switched off based
 * on the {@link DirectoryProperties#CACHE_ENABLED} attribute.
 * <p>
 * Note, the caching in this class refers to the DB caching of directory data, not of in memory caches of the
 * {@link RemoteDirectory} instances themselves
 *
 * @since 2.8
 */
public class ConfigurableDbCachingRemoteDirectoryInstanceLoader extends DbCachingRemoteDirectoryInstanceLoader {

    @SuppressWarnings("UnusedDeclaration")
    public ConfigurableDbCachingRemoteDirectoryInstanceLoader(DirectoryInstanceLoader remoteDirectoryInstanceLoader,
                                                              InternalDirectoryInstanceLoader internalDirectoryInstanceLoader,
                                                              DirectoryMonitorManager directoryMonitorManager,
                                                              DirectoryCacheFactory directoryCacheFactory) {
        super(remoteDirectoryInstanceLoader, internalDirectoryInstanceLoader, directoryMonitorManager,
                directoryCacheFactory);
    }

    @SuppressWarnings("UnusedDeclaration")
    public ConfigurableDbCachingRemoteDirectoryInstanceLoader(LDAPDirectoryInstanceLoader ldapDirectoryInstanceLoader,
                                                              RemoteCrowdDirectoryInstanceLoader remoteCrowdDirectoryInstanceLoader,
                                                              InternalDirectoryInstanceLoader internalDirectoryInstanceLoader,
                                                              DirectoryMonitorManager directoryMonitorManager,
                                                              DirectoryCacheFactory directoryCacheFactory) {
        super(ldapDirectoryInstanceLoader, remoteCrowdDirectoryInstanceLoader, internalDirectoryInstanceLoader,
                directoryMonitorManager, directoryCacheFactory);
    }


    @Override
    public RemoteDirectory getDirectory(Directory directory) throws DirectoryInstantiationException {
        // If Db caching is disabled for the directory, just return the remoteDirectory instance
        if (Boolean.parseBoolean(directory.getAttributes().get(DirectoryProperties.CACHE_ENABLED))) {
            return super.getDirectory(directory);
        } else {
            return getRawDirectory(directory.getId(), directory.getImplementationClass(), directory.getAttributes());
        }
    }
}
