package com.atlassian.crowd.directory.loader;

import java.util.Map;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheException;
import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.Supplier;
import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.event.directory.DirectoryDeletedEvent;
import com.atlassian.crowd.event.directory.DirectoryUpdatedEvent;
import com.atlassian.crowd.event.migration.XMLRestoreFinishedEvent;
import com.atlassian.crowd.exception.DirectoryInstantiationException;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;

import com.google.common.base.Throwables;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Caches the underlying remote directory base by id.
 * <p>
 * Note, this caching refers to holding the {@link RemoteDirectory} instances in memory, <em>not</em> the
 * db caching which may occur in underlying implementations
 *
 * @since v2.1
 */
public class CacheableDirectoryInstanceLoader implements DirectoryInstanceLoader {
    private static final Logger log = LoggerFactory.getLogger(CacheableDirectoryInstanceLoader.class);

    private final DelegatingDirectoryInstanceLoader delegate;
    private final EventPublisher eventPublisher;

    private final Cache<Long, RemoteDirectory> directoryCache;

    public CacheableDirectoryInstanceLoader(final DelegatingDirectoryInstanceLoader delegate,
                                            final EventPublisher eventPublisher,
                                            final CacheFactory cacheFactory) {
        this.delegate = delegate;
        this.eventPublisher = eventPublisher;
        this.directoryCache = createCache(cacheFactory);
        this.eventPublisher.register(this);
    }

    private Cache<Long, RemoteDirectory> createCache(CacheFactory cacheFactory) {
        return cacheFactory.getCache(CacheableDirectoryInstanceLoader.class.getName(), null,
                new CacheSettingsBuilder()
                        // RemoteDirectory is not serializable and invalidation needs to be cluster
                        // coherent so we explicitly configure replicateViaInvalidation aka hybrid
                        // caching
                        .remote()
                        .replicateViaInvalidation()
                        .build()

        );
    }

    public RemoteDirectory getDirectory(final Directory directory) throws DirectoryInstantiationException {
        // try to get the directory from the cache
        // We MUST only serve a single instance of DbCachingRemoteDirectory per directory ID because of
        // concurrency controls on the DbCachingRemoteDirectory sync refresh.
        final long id = directory.getId();
        try {
            return directoryCache.get(id, new Supplier<RemoteDirectory>() {
                @Override
                public RemoteDirectory get() {
                    try {
                        return delegate.getDirectory(directory);
                    } catch (DirectoryInstantiationException e) {
                        throw new CacheException(e);
                    }
                }
            });
        } catch (CacheException e) {
            Throwables.propagateIfInstanceOf(e.getCause(), DirectoryInstantiationException.class);
            throw new DirectoryInstantiationException(e);
        }
    }

    public RemoteDirectory getRawDirectory(final Long id, final String className, final Map<String, String> attributes)
            throws DirectoryInstantiationException {
        return delegate.getRawDirectory(id, className, attributes);
    }

    public boolean canLoad(final String className) {
        return delegate.canLoad(className);
    }

    // -----------------------------------------------------------------------------------------------------------------
    // EventListeners for refresh of directory cache
    // -----------------------------------------------------------------------------------------------------------------

    @EventListener
    public void handleEvent(DirectoryUpdatedEvent event) {
        // Remove the old directory instance from the cache
        directoryCache.remove(event.getDirectoryId());
    }

    @EventListener
    public void handleEvent(DirectoryDeletedEvent event) {
        // Remove the old directory instance from the cache
        directoryCache.remove(event.getDirectoryId());
    }

    @SuppressWarnings({"UnusedDeclaration"})
    @EventListener
    public void handleEvent(XMLRestoreFinishedEvent event) {
        // Clear all instances from the cache.
        directoryCache.removeAll();
        log.debug("Directory Cache cleared.");
    }

}
