package com.atlassian.crowd.manager.recovery;

import java.util.List;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.dao.application.ApplicationDAO;
import com.atlassian.crowd.directory.loader.DirectoryInstanceLoader;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.spi.DirectoryDao;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.manager.directory.BeforeGroupRemoval;
import com.atlassian.crowd.manager.directory.DirectoryManagerGeneric;
import com.atlassian.crowd.manager.directory.DirectorySynchroniser;
import com.atlassian.crowd.manager.directory.SynchronisationStatusManager;
import com.atlassian.crowd.manager.directory.monitor.poller.DirectoryPollerManager;
import com.atlassian.crowd.manager.permission.PermissionManager;
import com.atlassian.event.api.EventPublisher;

import com.google.common.collect.ImmutableList;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Overrides {@link DirectoryManagerGeneric} to provide recovery-mode aware behaviour.
 *
 * @since 2.7.2
 */
public class RecoveryModeAwareDirectoryManager extends DirectoryManagerGeneric {
    private final RecoveryModeService recoveryModeService;

    public RecoveryModeAwareDirectoryManager(DirectoryDao directoryDao, ApplicationDAO applicationDAO,
                                             EventPublisher eventPublisher, PermissionManager permissionManager,
                                             DirectoryInstanceLoader directoryInstanceLoader,
                                             DirectorySynchroniser directorySynchroniser,
                                             DirectoryPollerManager directoryPollerManager,
                                             ClusterLockService clusterLockService,
                                             SynchronisationStatusManager synchronisationStatusManager,
                                             BeforeGroupRemoval beforeGroupRemoval,
                                             RecoveryModeService recoveryModeService) {
        super(directoryDao, applicationDAO, eventPublisher, permissionManager, directoryInstanceLoader,
                directorySynchroniser, directoryPollerManager, clusterLockService, synchronisationStatusManager,
                beforeGroupRemoval);
        this.recoveryModeService = checkNotNull(recoveryModeService, "recoveryModeService");
    }

    @Override
    public Directory findDirectoryById(long directoryId) throws DirectoryNotFoundException {
        if (recoveryModeService.isRecoveryModeOn() && recoveryModeService.getRecoveryDirectory().getId().equals(directoryId)) {
            return recoveryModeService.getRecoveryDirectory();
        } else {
            return super.findDirectoryById(directoryId);
        }
    }

    @Override
    public List<Directory> findAllDirectories() {
        List<Directory> directories = super.findAllDirectories();
        if (recoveryModeService.isRecoveryModeOn()) {
            return ImmutableList.<Directory>builder()
                    .add(recoveryModeService.getRecoveryDirectory())
                    .addAll(directories)
                    .build();
        } else {
            return directories;
        }
    }

    @Override
    public Directory findDirectoryByName(String name) throws DirectoryNotFoundException {
        if (recoveryModeService.isRecoveryModeOn() && recoveryModeService.getRecoveryDirectory().getName().equalsIgnoreCase(name)) {
            return recoveryModeService.getRecoveryDirectory();
        } else {
            return super.findDirectoryByName(name);
        }
    }

}
