package com.atlassian.crowd.manager.directory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;

import javax.annotation.Nonnull;

import com.atlassian.crowd.directory.RemoteDirectory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.DirectoryEntity;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupType;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.search.Entity;
import com.atlassian.crowd.search.EntityDescriptor;
import com.atlassian.crowd.search.builder.QueryBuilder;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;
import com.atlassian.crowd.search.util.SearchResultsUtil;

import static com.atlassian.crowd.embedded.impl.IdentifierUtils.toLowerCase;
import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Searching infrastructure that properly belongs to {@code RemoteDirectory} but has historically been implemented
 * in {@code DirectoryManagerGeneric}.
 * <p>
 * This has been factored out in preparation for for migrating it to a more sensible home in the future.
 * </p>
 *
 * @since v2.8.0
 */
class RemoteDirectorySearcher {
    private final RemoteDirectory remoteDirectory;

    RemoteDirectorySearcher(@Nonnull RemoteDirectory remoteDirectory) {
        this.remoteDirectory = checkNotNull(remoteDirectory, "remoteDirectory");
    }

    boolean isUserDirectGroupMember(final String userName, final String groupName)
            throws OperationFailedException {
        return remoteDirectory.isUserDirectGroupMember(userName, groupName);
    }

    boolean isGroupDirectGroupMember(final String childGroup, final String parentGroup)
            throws OperationFailedException, DirectoryNotFoundException {
        return !childGroup.equals(parentGroup)
                && remoteDirectory.supportsNestedGroups()
                && remoteDirectory.isGroupDirectGroupMember(childGroup, parentGroup);
    }

    <T> List<T> searchDirectGroupRelationships(final MembershipQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        if (isNestedGroupQuery(query) && !remoteDirectory.supportsNestedGroups()) {
            return Collections.emptyList();
        }

        if (isLegacyQuery(query)) {
            return Collections.emptyList();
        }

        return remoteDirectory.searchGroupRelationships(query);
    }

    private static <T> boolean isNestedGroupQuery(MembershipQuery<T> query) {
        return query.getEntityToMatch().getEntityType() == Entity.GROUP
                && query.getEntityToReturn().getEntityType() == Entity.GROUP;
    }

    private static <T> boolean isLegacyQuery(MembershipQuery<T> query) {
        return isLegacyRole(query.getEntityToMatch()) || isLegacyRole(query.getEntityToReturn());
    }

    private static boolean isLegacyRole(EntityDescriptor entity) {
        return entity.getEntityType() == Entity.GROUP && entity.getGroupType() == GroupType.LEGACY_ROLE;
    }


    boolean isUserNestedGroupMember(final String username, final String groupName)
            throws OperationFailedException, DirectoryNotFoundException {
        if (remoteDirectory.supportsNestedGroups()) {
            return isUserNestedGroupMember(username, groupName, new HashSet<String>(64));
        }
        return isUserDirectGroupMember(username, groupName);
    }

    private boolean isUserNestedGroupMember(final String username, final String groupName,
                                            final Set<String> visitedGroups) throws OperationFailedException, DirectoryNotFoundException {
        if (!visitedGroups.add(toLowerCase(groupName))) {
            // Refuse to re-enter a cyclic group relationship
            return false;
        }

        // first check if the user is a direct member
        // if he's not a direct member, then check if he's a nested member of any of the subgroups (depth first)
        return remoteDirectory.isUserDirectGroupMember(username, groupName) ||
                isUserIndirectGroupMember(username, groupName, visitedGroups);
    }

    private boolean isUserIndirectGroupMember(final String username, final String groupName,
                                              final Set<String> visitedGroups) throws OperationFailedException, DirectoryNotFoundException {
        final List<Group> subGroups = searchDirectGroupRelationships(
                QueryBuilder.queryFor(Group.class, EntityDescriptor.group())
                        .childrenOf(EntityDescriptor.group())
                        .withName(groupName)
                        .returningAtMost(EntityQuery.ALL_RESULTS));

        for (Group childGroup : subGroups) {
            if (isUserNestedGroupMember(username, childGroup.getName(), visitedGroups)) {
                return true;
            }
        }
        return false;
    }


    boolean isGroupNestedGroupMember(final String childGroup, final String parentGroup)
            throws OperationFailedException, DirectoryNotFoundException {
        return !childGroup.equals(parentGroup)
                && remoteDirectory.supportsNestedGroups()
                && isGroupNestedGroupMember(childGroup, parentGroup, new HashSet<String>(64));
    }

    private boolean isGroupNestedGroupMember(final String childGroupName, final String parentGroupName,
                                             final Set<String> visitedGroups) throws OperationFailedException, DirectoryNotFoundException {
        if (!visitedGroups.add(toLowerCase(parentGroupName))) {
            // cycled around and still haven't been able to prove membership
            return false;
        }

        // first check if the child group is a direct member
        return remoteDirectory.isGroupDirectGroupMember(childGroupName, parentGroupName)
                || isGroupIndirectGroupMember(childGroupName, parentGroupName, visitedGroups);
    }

    private boolean isGroupIndirectGroupMember(final String childGroupName, final String parentGroupName,
                                               final Set<String> visitedGroups) throws OperationFailedException, DirectoryNotFoundException {
        // if it's not a direct member, then check if it's a nested member of any of the subgroups (depth first)
        final List<Group> subGroups = searchDirectGroupRelationships(
                QueryBuilder.queryFor(Group.class, EntityDescriptor.group())
                        .childrenOf(EntityDescriptor.group())
                        .withName(parentGroupName)
                        .returningAtMost(EntityQuery.ALL_RESULTS));

        for (Group childGroup : subGroups) {
            if (isGroupNestedGroupMember(childGroupName, childGroup.getName(), visitedGroups)) {
                return true;
            }
        }
        return false;
    }


    @SuppressWarnings("unchecked")
    <T> List<T> searchNestedGroupRelationships(final MembershipQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        if (!remoteDirectory.supportsNestedGroups()) {
            return searchDirectGroupRelationships(query);
        }

        List<? extends DirectoryEntity> relations = getRelatedDirectoryEntities(query);
        relations = SearchResultsUtil.constrainResults(relations, query.getStartIndex(), query.getMaxResults());

        //noinspection ObjectEquality
        if (query.getReturnType() == String.class) { // as name
            return (List<T>) SearchResultsUtil.convertEntitiesToNames(relations);
        }

        return (List<T>) relations;
    }

    private <T> List<? extends DirectoryEntity> getRelatedDirectoryEntities(final MembershipQuery<T> query)
            throws OperationFailedException, DirectoryNotFoundException {
        int totalResults = query.getStartIndex() + query.getMaxResults();
        if (query.getMaxResults() == EntityQuery.ALL_RESULTS) {
            totalResults = EntityQuery.ALL_RESULTS;
        }

        if (query.isFindChildren()) {
            return getChildDirectoryEntities(query, totalResults);
        }
        return getParentDirectoryEntities(query, totalResults);
    }

    private <T> List<? extends DirectoryEntity> getParentDirectoryEntities(final MembershipQuery<T> query,
                                                                           final int totalResults) throws OperationFailedException, DirectoryNotFoundException {
        // find memberships
        if (query.getEntityToReturn().getEntityType() != Entity.GROUP) {
            throw new IllegalArgumentException("You can only find the GROUP memberships of USER or GROUP");
        }

        if (query.getEntityToMatch().getEntityType() == Entity.USER) {
            // query is to find GROUP memberships of USER
            return findNestedGroupMembershipsOfUser(query.getEntityNameToMatch(),
                    query.getEntityToReturn().getGroupType(), totalResults);
        }

        if (query.getEntityToMatch().getEntityType() == Entity.GROUP) {
            // query is to find GROUP memberships of GROUP
            return findNestedGroupMembershipsOfGroup(query.getEntityNameToMatch(),
                    query.getEntityToReturn().getGroupType(), totalResults);
        }

        throw new IllegalArgumentException("You can only find the GROUP memberships of USER or GROUP");
    }

    private <T> List<? extends DirectoryEntity> getChildDirectoryEntities(final MembershipQuery<T> query,
                                                                          final int totalResults) throws OperationFailedException, DirectoryNotFoundException {
        if (query.getEntityToMatch().getEntityType() != Entity.GROUP) {
            throw new IllegalArgumentException("You can only find the GROUP or USER members of a GROUP");
        }

        if (query.getEntityToReturn().getEntityType() == Entity.USER) {
            // query is to find USER members of GROUP
            return findNestedUserMembersOfGroup(query.getEntityNameToMatch(),
                    query.getEntityToMatch().getGroupType(), totalResults);
        }

        if (query.getEntityToReturn().getEntityType() == Entity.GROUP) {
            // query is to find GROUP members of GROUP
            return findNestedGroupMembersOfGroup(query.getEntityNameToMatch(),
                    query.getEntityToMatch().getGroupType(), totalResults);
        }

        throw new IllegalArgumentException("You can only find the GROUP or USER members of a GROUP");
    }

    private List<Group> findNestedGroupMembershipsOfGroup(final String groupName, final GroupType groupType,
                                                          final int maxResults) throws OperationFailedException, DirectoryNotFoundException {
        final Group group;
        try {
            group = remoteDirectory.findGroupByName(groupName);
        } catch (GroupNotFoundException e) {
            return Collections.emptyList();
        }

        return findNestedGroupMembershipsIncludingGroups(Arrays.asList(group), groupType, maxResults, false);
    }

    private List<Group> findNestedGroupMembershipsIncludingGroups(final List<Group> groups, final GroupType groupType,
                                                                  final int maxResults, boolean includeOriginal) throws OperationFailedException, DirectoryNotFoundException {
        Queue<Group> groupsToVisit = new LinkedList<Group>();
        Set<Group> nestedParents = new LinkedHashSet<Group>();

        groupsToVisit.addAll(groups);

        // Should the original groups be included in the results?
        int totalResults = maxResults;
        if (maxResults != EntityQuery.ALL_RESULTS && !includeOriginal) {
            totalResults = maxResults + groups.size();
        }

        // now find the nested parents of the direct group memberships (similar to findNestedGroupMembershipsOfGroup)
        // keep iterating while there are more groups to explore AND (we are searching for everything OR we haven't found enough results)
        while (!groupsToVisit.isEmpty() && (totalResults == EntityQuery.ALL_RESULTS || nestedParents
                .size() < totalResults)) {
            Group groupToVisit = groupsToVisit.remove();

            // avoid cycles
            if (!nestedParents.contains(groupToVisit)) {
                // add this group to nested parents
                nestedParents.add(groupToVisit);

                // find direct parent groups
                List<Group> directParents = searchDirectGroupRelationships(
                        QueryBuilder.queryFor(Group.class, EntityDescriptor.group(groupType))
                                .parentsOf(EntityDescriptor.group(groupType))
                                .withName(groupToVisit.getName())
                                .returningAtMost(maxResults));

                // visit them later
                groupsToVisit.addAll(directParents);
            }
        }

        if (!includeOriginal) {
            nestedParents.removeAll(groups);
        }

        return new ArrayList<Group>(nestedParents);
    }

    private List<Group> findNestedGroupMembershipsOfUser(final String username, final GroupType groupType,
                                                         final int maxResults) throws OperationFailedException, DirectoryNotFoundException {
        List<Group> directGroupMemberships = searchDirectGroupRelationships(
                QueryBuilder.queryFor(Group.class, EntityDescriptor.group(groupType))
                        .parentsOf(EntityDescriptor.user())
                        .withName(username)
                        .returningAtMost(maxResults));

        return findNestedGroupMembershipsIncludingGroups(directGroupMemberships, groupType, maxResults, true);
    }

    private List<Group> findNestedGroupMembersOfGroup(final String groupName, final GroupType groupType,
                                                      final int maxResults) throws OperationFailedException, DirectoryNotFoundException {
        Group group;
        try {
            group = remoteDirectory.findGroupByName(groupName);
        } catch (GroupNotFoundException e) {
            return Collections.emptyList();
        }

        Queue<Group> groupsToVisit = new LinkedList<Group>();
        Set<Group> nestedMembers = new LinkedHashSet<Group>();

        groupsToVisit.add(group);

        // keep iterating while there are more groups to explore AND (we are searching for everything OR we haven't found enough results)
        while (!groupsToVisit.isEmpty() && (maxResults == EntityQuery.ALL_RESULTS || nestedMembers
                .size() < maxResults + 1)) {
            Group groupToVisit = groupsToVisit.remove();

            // avoid cycles
            if (!nestedMembers.contains(groupToVisit)) {
                // add this group to nested members
                nestedMembers.add(groupToVisit);

                // find direct subgroups
                List<Group> directMembers = searchDirectGroupRelationships(
                        QueryBuilder.queryFor(Group.class, EntityDescriptor.group(groupType))
                                .childrenOf(EntityDescriptor.group(groupType))
                                .withName(groupToVisit.getName())
                                .returningAtMost(maxResults));

                // visit them later
                groupsToVisit.addAll(directMembers);
            }
        }

        // remove the original group we are finding the members of (this will be in the nested members set to prevent cycles)
        nestedMembers.remove(group);

        return new ArrayList<Group>(nestedMembers);
    }

    private List<User> findNestedUserMembersOfGroup(final String groupName, final GroupType groupType,
                                                    final int maxResults) throws OperationFailedException, DirectoryNotFoundException {
        final Group group;
        try {
            group = remoteDirectory.findGroupByName(groupName);
        } catch (GroupNotFoundException e) {
            return Collections.emptyList();
        }

        final Queue<Group> groupsToVisit = new LinkedList<Group>();
        final Set<Group> nestedGroupMembers = new LinkedHashSet<Group>();
        final Set<User> nestedUserMembers = new LinkedHashSet<User>();

        groupsToVisit.add(group);

        // keep iterating while there are more groups to explore AND (we are searching for everything OR we haven't found enough results)
        while (!groupsToVisit.isEmpty() && (maxResults == EntityQuery.ALL_RESULTS || nestedUserMembers
                .size() < maxResults)) {
            final Group groupToVisit = groupsToVisit.remove();

            final List<User> directUserMembers = searchDirectGroupRelationships(
                    QueryBuilder.queryFor(User.class, EntityDescriptor.user())
                            .childrenOf(EntityDescriptor.group(groupType))
                            .withName(groupToVisit.getName())
                            .returningAtMost(maxResults));
            nestedUserMembers.addAll(directUserMembers);

            // avoid cycles
            if (!nestedGroupMembers.contains(groupToVisit)) {
                // add this group to nested members
                nestedGroupMembers.add(groupToVisit);

                // find direct subgroups
                final List<Group> directGroupMembers = searchDirectGroupRelationships(
                        QueryBuilder.queryFor(Group.class, EntityDescriptor.group(groupType))
                                .childrenOf(EntityDescriptor.group(groupType))
                                .withName(groupToVisit.getName())
                                .returningAtMost(EntityQuery.ALL_RESULTS));

                // visit them later
                groupsToVisit.addAll(directGroupMembers);
            }
        }

        return new ArrayList<User>(nestedUserMembers);
    }
}


