package com.atlassian.crowd.manager.directory;

import java.util.concurrent.locks.Lock;

import com.atlassian.beehive.ClusterLockService;
import com.atlassian.crowd.directory.SynchronisableDirectory;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisationFailedEvent;
import com.atlassian.crowd.event.directory.RemoteDirectorySynchronisedEvent;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.event.api.EventPublisher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * An implementation of a {@link DirectorySynchroniser}.
 */
public class DirectorySynchroniserImpl implements DirectorySynchroniser {
    private final static Logger log = LoggerFactory.getLogger(DirectorySynchroniser.class);
    private final ClusterLockService lockService;
    private final DirectorySynchroniserHelper directorySynchroniserHelper;
    private final SynchronisationStatusManager synchronisationStatusManager;
    private final EventPublisher eventPublisher;

    public DirectorySynchroniserImpl(ClusterLockService lockService,
                                     DirectorySynchroniserHelper directorySynchroniserHelper,
                                     SynchronisationStatusManager synchronisationStatusManager, EventPublisher eventPublisher) {
        this.lockService = lockService;
        this.directorySynchroniserHelper = directorySynchroniserHelper;
        this.synchronisationStatusManager = synchronisationStatusManager;
        this.eventPublisher = eventPublisher;
    }

    /**
     * Synchronises a remote directory. For performance reasons, this method must not run within
     * a transaction. As a synchronisation may involve a huge amount of data, individual batches
     * are run in their own transactions. Running overall synchronisation in a transaction would defeat
     * this and is prevented.
     */
    @Transactional(propagation = Propagation.NEVER)
    @Override
    public void synchronise(final SynchronisableDirectory remoteDirectory, final SynchronisationMode mode)
            throws DirectoryNotFoundException, OperationFailedException {
        long directoryId = remoteDirectory.getDirectoryId();
        final Directory directory = findDirectoryById(directoryId);
        if (!directory.isActive()) {
            log.debug("Request to synchronise directory [ {} ] in {} mode is returning silently because the directory is not active.",
                    directoryId, mode);
            return;
        }

        log.debug("request to synchronise directory [ {} ] in {} mode", directoryId, mode);

        final Lock lock = lockService.getLockForName(DirectorySynchronisationUtils.getLockName(directoryId));

        if (lock.tryLock()) {
            boolean successful = true;
            final long startTime = System.currentTimeMillis();
            try {
                directorySynchroniserHelper.updateSyncStartTime(remoteDirectory);
                synchronisationStatusManager.syncStarted(directory);
                try {
                    remoteDirectory.synchroniseCache(mode, synchronisationStatusManager);
                } catch (Exception ignored) {
                    successful = false;
                    throw ignored;
                } finally {
                    directorySynchroniserHelper.updateSyncEndTime(remoteDirectory);
                    synchronisationStatusManager.syncFinished(directoryId);
                }
            } finally {
                lock.unlock();

                final long timeTakenInMs = System.currentTimeMillis() - startTime;
                if (successful) {
                    eventPublisher.publish(new RemoteDirectorySynchronisedEvent(this, remoteDirectory, timeTakenInMs));
                } else {
                    eventPublisher.publish(new RemoteDirectorySynchronisationFailedEvent(this, remoteDirectory, timeTakenInMs));
                }
            }
        } else {
            log.debug("directory [ {} ] already synchronising", directoryId);
        }
    }

    public boolean isSynchronising(long directoryId) throws DirectoryNotFoundException {
        return directorySynchroniserHelper.isSynchronising(directoryId);
    }

    private Directory findDirectoryById(final long directoryId) throws DirectoryNotFoundException {
        return directorySynchroniserHelper.findDirectoryById(directoryId);
    }
}
