package com.atlassian.crowd.manager.cache;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheEntryListener;
import com.atlassian.cache.Supplier;

import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * A {@link Cache} backed by Ehcache. This is implemented
 * separately from <code>atlassian-cache-ehcache</code> to
 * intentionally omit API that may not be supported by a future simpler version of
 * {@link Cache}.
 */
class EhcacheBackedCache<K, V> implements Cache<K, V> {
    private final Ehcache ehcache;

    EhcacheBackedCache(Ehcache ehcache) {
        this.ehcache = checkNotNull(ehcache);
    }

    @Override
    public boolean containsKey(@Nonnull K key) {
        return ehcache.get(key) != null;
    }

    private static <V> V contentsOrNull(Element e) {
        return (e != null) ? (V) e.getObjectValue() : null;
    }

    @Override
    @Nullable
    public V get(@Nonnull K key) {
        return contentsOrNull(ehcache.get(key));
    }

    @Override
    @Nonnull
    public V get(@Nonnull K key, @Nonnull Supplier<? extends V> valueSupplier) {
        V val = get(key);
        if (val == null) {
            val = valueSupplier.get();
            put(key, val);
        }
        return val;
    }

    @Override
    @Nonnull
    public String getName() {
        return ehcache.getName();
    }

    @Override
    public void put(@Nonnull K key, @Nonnull V value) {
        ehcache.put(new Element(key, value));
    }

    @Override
    @Nullable
    public V putIfAbsent(@Nonnull K key, @Nonnull V value) {
        Element current = ehcache.putIfAbsent(new Element(key, value));

        return contentsOrNull(current);
    }

    @Override
    public void remove(@Nonnull K key) {
        ehcache.remove(key);
    }

    @Override
    public boolean remove(@Nonnull K key, @Nonnull V value) {
        return ehcache.removeElement(new Element(key, value));
    }

    @Override
    public void removeAll() {
        ehcache.removeAll();
    }

    @Override
    public boolean replace(@Nonnull K key, @Nonnull V oldValue, @Nonnull V newValue) {
        return ehcache.replace(new Element(key, oldValue), new Element(key, newValue));
    }

    // Expensive; may need to be removed
    @Override
    @Nonnull
    public Collection<K> getKeys() {
        return ehcache.getKeys();
    }

    // Not implemented. Will not be implemented.
    @Override
    public void addListener(@Nonnull CacheEntryListener<K, V> listener, boolean includeValues) {
        throw new UnsupportedOperationException();
    }

    // Not implemented. Will not be implemented.
    @Override
    public void removeListener(@Nonnull CacheEntryListener<K, V> listener) {
        throw new UnsupportedOperationException();
    }
}
