package com.atlassian.crowd.manager.cache;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheFactory;
import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.Supplier;

import net.sf.ehcache.Ehcache;

/**
 * A {@link CacheFactory} (and also a {@link CacheManager}) backed by Ehcache. This is implemented
 * separately from <code>atlassian-cache-ehcache</code> to restrict the scope to how Crowd currently
 * uses Ehcache, and intentionally omit API that may not be supported by a future simpler version of
 * {@link Cache}.
 */
public class CacheFactoryEhcache implements CacheFactory, CacheManager {
    private final net.sf.ehcache.CacheManager cacheManager;

    public CacheFactoryEhcache(net.sf.ehcache.CacheManager cacheManager) {
        this.cacheManager = cacheManager;
    }

    @Override
    @Nonnull
    public <K, V> Cache<K, V> getCache(@Nonnull String name) {
        Ehcache ehcache = cacheManager.addCacheIfAbsent(name);

        return new EhcacheBackedCache<K, V>(ehcache);
    }

    @Override
    @Nonnull
    public <K, V> Cache<K, V> getCache(@Nonnull Class<?> owningClass, @Nonnull String name) {
        return getCache(owningClass.getName() + "/" + name);
    }

    @Override
    @Nonnull
    public <K, V> Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Nonnull
    public <K, V> Cache<K, V> getCache(@Nonnull String name, @Nullable CacheLoader<K, V> loader,
                                       @Nonnull CacheSettings required) {
        // WebResourceSetCache calls this; support enough to keep it happy
        if (loader == null) {
            // Ignore the CacheSettings
            return getCache(name);
        } else {
            throw new UnsupportedOperationException();
        }
    }

    @Override
    @Nonnull
    public <K, V> Cache<K, V> getCache(@Nonnull String name, @Nonnull Class<K> keyType, @Nonnull Class<V> valueType) {
        return getCache(name);
    }

    // Not implemented. Will not be implemented.
    @Override
    @Nonnull
    public <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass, @Nonnull String name,
                                                     @Nonnull Supplier<V> supplier) {
        throw new UnsupportedOperationException();
    }

    // Not implemented. Will not be implemented.
    @Override
    @Nonnull
    public <V> CachedReference<V> getCachedReference(@Nonnull Class<?> owningClass, @Nonnull String name,
                                                     @Nonnull Supplier<V> supplier, @Nonnull CacheSettings required) {
        throw new UnsupportedOperationException();
    }

    // Not implemented. Will not be implemented.
    @Override
    @Nonnull
    public <V> CachedReference<V> getCachedReference(@Nonnull String name, @Nonnull Supplier<V> supplier) {
        throw new UnsupportedOperationException();
    }

    // Not implemented. Will not be implemented.
    @Override
    @Nonnull
    public <V> CachedReference<V> getCachedReference(@Nonnull String name, @Nonnull Supplier<V> supplier,
                                                     @Nonnull CacheSettings required) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void flushCaches() {
        cacheManager.clearAll();
    }

    @Override
    @Nonnull
    public Collection<Cache<?, ?>> getCaches() {
        throw new UnsupportedOperationException();
    }

    @Override
    @Nullable
    public ManagedCache getManagedCache(@Nonnull String name) {
        throw new UnsupportedOperationException();
    }

    @Override
    @Nonnull
    public Collection<ManagedCache> getManagedCaches() {
        throw new UnsupportedOperationException();
    }

    @Override
    public void shutdown() {
        cacheManager.shutdown();
    }
}
