package com.atlassian.crowd.manager.application;

import java.util.Collections;
import java.util.List;

import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.manager.directory.DirectoryManager;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.query.membership.MembershipQuery;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link SearchStrategy} which optimizes for a single directory
 *
 * @since 2.9
 */
public class SingleDirectorySearchStrategy implements SearchStrategy {
    private static final Logger logger = LoggerFactory.getLogger(SingleDirectorySearchStrategy.class);

    private final DirectoryManager directoryManager;
    private final long directoryId;

    public SingleDirectorySearchStrategy(DirectoryManager directoryManager, long directoryId) {
        this.directoryManager = directoryManager;
        this.directoryId = directoryId;
    }

    @Override
    public <T> List<T> searchUsers(EntityQuery<T> query) {
        try {
            return directoryManager.searchUsers(directoryId, query);
        } catch (DirectoryNotFoundException e) {
            // directory does not exist, just skip
        } catch (OperationFailedException e) {
            logger.error("Failed to search underlying directory", e);
        }
        return Collections.emptyList();
    }

    @Override
    public <T> List<T> searchGroups(EntityQuery<T> query) {
        try {
            return directoryManager.searchGroups(directoryId, query);
        } catch (DirectoryNotFoundException e) {
            // directory does not exist, just skip
        } catch (OperationFailedException e) {
            logger.error("Failed to search underlying directory", e);
        }
        return Collections.emptyList();
    }

    @Override
    public <T> List<T> searchDirectGroupRelationships(MembershipQuery<T> query) {
        try {
            return directoryManager.searchDirectGroupRelationships(directoryId, query);
        } catch (DirectoryNotFoundException e) {
            // directory does not exist, just skip
        } catch (OperationFailedException e) {
            logger.error("Failed to search underlying directory", e);
        }
        return Collections.emptyList();
    }

    @Override
    public <T> List<T> searchNestedGroupRelationships(MembershipQuery<T> query) {
        try {
            return directoryManager.searchNestedGroupRelationships(directoryId, query);
        } catch (DirectoryNotFoundException e) {
            // directory does not exist, just skip
        } catch (OperationFailedException e) {
            logger.error("Failed to search underlying directory", e);
        }

        return Collections.emptyList();
    }
}
