package com.atlassian.crowd.manager.application;

import java.util.Objects;

import com.atlassian.crowd.cache.UserAuthorisationCache;
import com.atlassian.crowd.exception.ApplicationPermissionException;
import com.atlassian.crowd.exception.InactiveAccountException;
import com.atlassian.crowd.exception.InvalidUserException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.user.User;

import org.springframework.transaction.annotation.Transactional;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Implementation of ApplicationService which caches the result of ApplicationService methods.
 *
 * @since v2.2
 */
@Transactional
public class CachingApplicationService extends AbstractDelegatingApplicationService {
    private final UserAuthorisationCache userAuthorisationCache;

    public CachingApplicationService(final ApplicationService applicationService, final UserAuthorisationCache userAuthorisationCache) {
        super(applicationService);
        this.userAuthorisationCache = checkNotNull(userAuthorisationCache);
    }

    public boolean isUserAuthorised(final Application application, final String username) {
        try {
            final User user = findUserByName(application, username);
            return isUserAuthorised(application, user);
        } catch (UserNotFoundException e) {
            return false;
        }
    }

    @Override
    public boolean isUserAuthorised(final Application application, final User user) {
        final Boolean allowedToAuthenticate = userAuthorisationCache.isPermitted(user.getName(), application.getName());
        if (allowedToAuthenticate != null) {
            return allowedToAuthenticate;
        } else {
            final boolean permitted = getApplicationService().isUserAuthorised(application, user);
            // only cache positive results
            if (permitted) {
                userAuthorisationCache.setPermitted(user.getName(), application.getName(), permitted);
            }
            return permitted;
        }
    }

    public User renameUser(Application application, String oldUserName, String newUsername) throws UserNotFoundException, OperationFailedException, ApplicationPermissionException, InvalidUserException {
        try {
            return getApplicationService().renameUser(application, oldUserName, newUsername);
        } finally {
            userAuthorisationCache.clear(oldUserName, application.getName());
            userAuthorisationCache.clear(newUsername, application.getName());
        }
    }

    public void removeUser(final Application application, final String user)
            throws OperationFailedException, UserNotFoundException, ApplicationPermissionException {
        try {
            getApplicationService().removeUser(application, user);
        } finally {
            userAuthorisationCache.clear(user, application.getName());
        }
    }

    @Override
    public User userAuthenticated(Application application, String username) throws UserNotFoundException, OperationFailedException, InactiveAccountException {
        userAuthorisationCache.clear(username, application.getName());
        final User updatedUser = getApplicationService().userAuthenticated(application, username);
        if (!Objects.equals(username, updatedUser.getName())) {
            userAuthorisationCache.clear(updatedUser.getName(), application.getName());
        }
        return updatedUser;
    }
}
