package com.atlassian.crowd.core.event;

import java.util.Collection;

import com.atlassian.event.api.EventPublisher;

import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

/**
 * Delays publishing any events if a transaction is active, until that transaction commits.
 * Events not published in transaction are unaffected. If the active transaction is rolled back, the events are not published.
 *
 * Note that any transactional operations done in afterCommit(), need to be wrapped in a PROPAGATION_REQUIRES_NEW transaction,
 * (see {@link org.springframework.transaction.support.TransactionSynchronization#afterCommit()}). This is guaranteed by
 * {@link TransactionAwareEventDispatcher}
 *
 * Since publishing an event during a transaction causes a {@link org.springframework.transaction.support.TransactionSynchronization}
 * to be registered for cases where many events are sent at once, it makes sense to use {@link #publishAll(Collection)}, to wrap them
 * in a single TransactionSynchronization, as transaction performance degrades notably with the number of distinct synchronizations registered.
 */
public class TransactionAwareEventPublisher extends DelegatingMultiEventPublisher implements EventPublisher, MultiEventPublisher {
    TransactionAwareEventPublisher(EventPublisher delegate) {
        super(delegate);
    }

    @Override
    public void publish(Object event) {
        if (shouldPostponeEvent()) {
            TransactionSynchronizationManager.registerSynchronization(createSynchronization(event));
        } else {
            delegate.publish(event);
        }
    }

    @Override
    public void publishAll(Collection<Object> events) {
        if (shouldPostponeEvent()) {
            TransactionSynchronizationManager.registerSynchronization(createSynchronization(events));
        } else {
            events.forEach(delegate::publish);
        }
    }

    private boolean shouldPostponeEvent() {
        return TransactionSynchronizationManager.isActualTransactionActive() && TransactionSynchronizationManager.isSynchronizationActive();
    }

    private TransactionSynchronizationAdapter createSynchronization(final Object event) {
        return new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                delegate.publish(event);
            }
        };
    }

    private TransactionSynchronizationAdapter createSynchronization(final Collection<Object> events) {
        return new TransactionSynchronizationAdapter() {
            @Override
            public void afterCommit() {
                events.forEach(delegate::publish);
            }
        };
    }
}
