package com.atlassian.crowd.core.tiny;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

public class DateUtils {
    private static final Logger log = LoggerFactory.getLogger(DateUtils.class);

    private static final long MINUTE_SECONDS = TimeUnit.MINUTES.toSeconds(1);
    private static final long HOUR_SECONDS = TimeUnit.HOURS.toSeconds(1);
    private static final long DAY_SECONDS = TimeUnit.DAYS.toSeconds(1);
    private static final long WEEK_SECONDS = 7 * DAY_SECONDS;

    // This is used by the Velocity templates as a bean
    private final ResourceBundle resourceBundle;

    public DateUtils(ResourceBundle resourceBundle) {
        this.resourceBundle = resourceBundle;
    }

    /**
     * Converts a number of seconds into a pretty formatted data string.  The resolution is in minutes.  So if the number of seconds is greater than a minute, it will
     * only be shown down top minute resolution.  If the number of seconds is less than a minute it will be shown in seconds.
     * <p/>
     * So for example <code>76</code> becomes <code>'1 minute'</code>, while <code>42</code> becomes <code>'42 seconds'</code>
     *
     * @param numSecs        the number of seconds in the duration
     * @param resourceBundle a resouce bundle for i18n
     * @return a string in readable pretty duration format, using minute resolution
     */
    static String getDurationPretty(long numSecs, ResourceBundle resourceBundle) {
        return getDurationPrettySeconds(numSecs, DAY_SECONDS, WEEK_SECONDS, resourceBundle::getString, false);
    }

    /**
     * Converts a number of seconds into a pretty formatted data string.  The resolution is in seconds.
     * <p>
     * So for example <code>76</code> becomes <code>'1 minute, 16 seconds'</code>, while <code>42</code> becomes <code>'42 seconds'</code>
     *
     * @param numSecs        the number of seconds in the duration
     * @param resourceBundle a resouce bundle for i18n
     * @return a string in readable pretty duration format, using second resolution
     */
    public static String getDurationPrettySecondsResolution(long numSecs, ResourceBundle resourceBundle) {
        return getDurationPrettySeconds(numSecs, DAY_SECONDS, WEEK_SECONDS, resourceBundle::getString, true);
    }

    public static String getDurationPretty(long numSecs, Function<String, String> i18nResolver, boolean secondsResolution) {
        return getDurationPrettySeconds(numSecs, DAY_SECONDS, WEEK_SECONDS, i18nResolver, secondsResolution);
    }

    /**
     * Get a pretty formatted duration for the given number of seconds. (e.g. "4 days, 2 hours, 30 minutes")
     *
     * @param numSecs         the number of seconds in the duration
     * @param secondsPerDay   the number of seconds in a "day"
     * @param secondsPerWeek  the number of seconds in a "week"
     * @param resourceBundle  the bundle containing translations for the strings used in the pretty string (e.g. "days")
     * @param secondsDuration if false only display down to the minute even if there are some seconds, else display seconds
     * @return the formatted pretty duration
     */
    static String getDurationPrettySeconds(long numSecs, long secondsPerDay, long secondsPerWeek, Function<String, String> resourceBundle, boolean secondsDuration) {
        // use perWeek to calculate perYear because that already has "days per week" already figured in. if a week only had 3 days, for instance then
        // doing secondsPerDay * 365 would overestimate how much we can get done in a year.
        final long secondsPerYear = secondsPerWeek * 52;
        return getDurationPrettySeconds(numSecs, secondsPerYear, secondsPerDay, secondsPerWeek, resourceBundle, secondsDuration);
    }

    /*
     * This implementation method returns things in "minute resolution" unless the secondResolution flag is true
     */
    private static String getDurationPrettySeconds(long numSecs, long secondsPerYear, long secondsPerDay, long secondsPerWeek, Function<String, String> resourceBundle, boolean secondResolution) {
        if (numSecs == 0) {
            if (secondResolution) {
                return "0 " + getText(resourceBundle, "core.dateutils.seconds");
            } else {
                return "0 " + getText(resourceBundle, "core.dateutils.minutes");
            }
        }

        StringBuilder result = new StringBuilder();

        if (numSecs >= secondsPerYear) {
            long years = numSecs / secondsPerYear;
            result.append(years).append(' ');

            if (years > 1) {
                result.append(getText(resourceBundle, "core.dateutils.years"));
            } else {
                result.append(getText(resourceBundle, "core.dateutils.year"));
            }

            result.append(", ");
            numSecs = numSecs % secondsPerYear;
        }

        if (numSecs >= secondsPerWeek) {
            long weeks = numSecs / secondsPerWeek;
            result.append(weeks).append(' ');

            if (weeks > 1) {
                result.append(getText(resourceBundle, "core.dateutils.weeks"));
            } else {
                result.append(getText(resourceBundle, "core.dateutils.week"));
            }

            result.append(", ");
            numSecs = numSecs % secondsPerWeek;
        }

        if (numSecs >= secondsPerDay) {
            long days = numSecs / secondsPerDay;
            result.append(days).append(' ');

            if (days > 1) {
                result.append(getText(resourceBundle, "core.dateutils.days"));
            } else {
                result.append(getText(resourceBundle, "core.dateutils.day"));
            }

            result.append(", ");
            numSecs = numSecs % secondsPerDay;
        }

        if (numSecs >= HOUR_SECONDS) {
            long hours = numSecs / HOUR_SECONDS;
            result.append(hours).append(' ');

            if (hours > 1) {
                result.append(getText(resourceBundle, "core.dateutils.hours"));
            } else {
                result.append(getText(resourceBundle, "core.dateutils.hour"));
            }

            result.append(", ");
            numSecs = numSecs % HOUR_SECONDS;
        }

        if (numSecs >= MINUTE_SECONDS) {
            long minute = numSecs / MINUTE_SECONDS;
            result.append(minute).append(' ');


            if (minute > 1) {
                result.append(getText(resourceBundle, "core.dateutils.minutes"));
            } else {
                result.append(getText(resourceBundle, "core.dateutils.minute"));
            }

            result.append(", ");

            // if we want seconds resolution we need to reduce it down to seconds here
            if (secondResolution) {
                numSecs = numSecs % MINUTE_SECONDS;
            }
        }

        if (numSecs >= 1 && numSecs < MINUTE_SECONDS) {
            result.append(numSecs).append(' ');


            if (numSecs > 1) {
                result.append(getText(resourceBundle, "core.dateutils.seconds"));
            } else {
                result.append(getText(resourceBundle, "core.dateutils.second"));
            }

            result.append(", ");
        }

        if (result.length() > 2) {
            // remove the ", " on the end
            return result.substring(0, result.length() - 2);
        } else {
            return result.toString();
        }
    }

    /**
     * This is used by the Velocity templates as a bean
     *
     * @param l a duration in seconds
     * @return a pretty formatted version of the duration
     */
    public String formatDurationPretty(long l) {
        return DateUtils.getDurationPretty(l, resourceBundle);
    }

    private static String getText(Function<String, String> i18nResolver, String key) {
        try {
            return i18nResolver.apply(key);
        } catch (MissingResourceException e) {
            log.error("Key not found in bundle", e);
            return "";
        }
    }
}
