package com.atlassian.crowd.common.util;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Simple utility class to record execution time.
 */
public class TimedCalls {
    private static ThreadLocal<TimedCalls> timedCalls = new ThreadLocal<>();

    private static class Stats {
        long timeMs = 0;
        int count = 0;

        public void add(long durationMs) {
            timeMs += durationMs;
            count++;
        }

        public String describe(long totalTimeMs) {
            return "total: " + timeMs
                    + " count: " + count
                    + " avg: " + (timeMs / count)
                    + " percentage: " + (timeMs * 100 / totalTimeMs);
        }
    }

    private final long startTime = System.currentTimeMillis();
    private final Map<String, Stats> durations = new LinkedHashMap<>();
    private String currentTimer;
    private long currentTimerStartMs;

    /**
     * Records duration with given name.
     */
    public void addDuration(String name, long duration) {
        durations.computeIfAbsent(name, ignore -> new Stats()).add(duration);
    }

    /**
     * Starts new timer. Finishes the previous timer, if not yet finished.
     */
    public void startTimer(String name) {
        finishTimerIfPresent();
        currentTimer = name;
        currentTimerStartMs = System.currentTimeMillis();
    }

    /**
     * Finishes timer and logs duration.
     */
    public void finishTimer() {
        addDuration(currentTimer, System.currentTimeMillis() - currentTimerStartMs);
        currentTimer = null;
    }

    private void finishTimerIfPresent() {
        if (currentTimer != null) {
            finishTimer();
        }
    }

    /**
     * Returns map of timer name to it's total duration.
     */
    public Map<String, Long> createDurationMap() {
        finishTimerIfPresent();
        return ImmutableMap.copyOf(Maps.transformValues(durations, s -> s.timeMs));
    }

    /**
     * Returns string representations of longest timers.
     */
    public List<String> getTopTimersToString(int count) {
        return getTopTimersToString(System.currentTimeMillis() - startTime, count);
    }

    /**
     * Returns string representations of longest timers.
     */
    public List<String> getTopTimersToString(long totalTimeMs, int count) {
        return durations.entrySet().stream()
                .sorted(Comparator.comparing((Map.Entry<String, Stats> entry) -> entry.getValue().timeMs).reversed())
                .limit(count)
                .map(entry -> entry.getKey() + ": " + entry.getValue().describe(totalTimeMs))
                .collect(Collectors.toList());
    }

    /**
     * Returns instance attached to the current thread.
     */
    public static TimedCalls getInstance() {
        return timedCalls.get();
    }

    /**
     * Detaches and returns instance for the current thread.
     */
    public static TimedCalls clearInstance() {
        TimedCalls result = timedCalls.get();
        timedCalls.remove();
        return result;
    }

    /**
     * Creates new instance and attaches it to the current thread.
     */
    public static TimedCalls createInstanceForCurrentThread() {
        TimedCalls result = new TimedCalls();
        timedCalls.set(result);
        return result;
    }
}
