package com.atlassian.crowd.common.util;

import org.apache.commons.lang3.ClassUtils;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

/**
 * Utility class simplifying creation of proxy objects.
 */
public class ProxyUtil {
    private ProxyUtil() {
    }

    /**
     * Creates a proxy that will call {@code service} with provided {@code classLoader}. After the call original
     * context class loader will be restored.
     * @see Thread#setContextClassLoader(ClassLoader)
     */
    public static <T> T runWithContextClassLoader(ClassLoader classLoader, T service) {
        return newProxy(service, (Object proxy, Method method, Object[] args) -> {
            Thread currentThread = Thread.currentThread();
            ClassLoader original = currentThread.getContextClassLoader();
            try {
                currentThread.setContextClassLoader(classLoader);
                return method.invoke(service, args);
            } catch (InvocationTargetException e) {
                throw e.getCause();
            } catch (IllegalAccessException e) {
                throw new RuntimeException(e);
            } finally {
                currentThread.setContextClassLoader(original);
            }
        });
    }

    public static <T> T newProxy(T service, InvocationHandler handler) {
        return (T) Proxy.newProxyInstance(
                service.getClass().getClassLoader(),
                ClassUtils.getAllInterfaces(service.getClass()).toArray(new Class[0]), handler);
    }
}
