package com.atlassian.crowd.common.properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.time.temporal.TemporalUnit;
import java.util.Optional;

public class DurationSystemProperty extends AbstractSystemProperty<Duration> {
    private static final Logger logger = LoggerFactory.getLogger(DurationSystemProperty.class);

    private final TemporalUnit temporalUnit;

    public DurationSystemProperty(@Nonnull String propertyName, @Nonnull TemporalUnit temporalUnit, long defaultValue) {
        super(propertyName, Duration.of(defaultValue, temporalUnit));
        this.temporalUnit = temporalUnit;
    }

    @Nonnull
    @Override
    public Duration getValue() {
        final String rawValue = System.getProperty(propertyName);
        try {
            return Optional.ofNullable(rawValue)
                    .map(Long::parseLong)
                    .map(value -> Duration.of(value, temporalUnit))
                    .orElse(defaultValue);
        } catch (NumberFormatException e) {
            logger.warn("Illegal value of system property " + propertyName + ", expected a long integer but was: " + rawValue);
            return defaultValue;
        }
    }

    @Override
    public void setValue(@Nonnull Duration value) {
        System.setProperty(propertyName, String.valueOf(value.toNanos() / temporalUnit.getDuration().toNanos()));
    }
}
