package com.atlassian.crowd.model.group;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

/**
 * Represents a new or modified group for incremental synchronisation. In addition to the group properties this class
 * also contains identifiers of users and groups that have joined the underlying group or left it. This is particularly
 * suitable for directories that provide a list of new and removed memberships for changed groups, for example Azure
 * Active Directory.
 */
public class GroupWithMembershipChanges implements Group {
    private final ImmutableGroup group;
    private final Set<String> userChildrenIdsToAdd;
    private final Set<String> userChildrenIdsToDelete;
    private final Set<String> groupChildrenIdsToAdd;
    private final Set<String> groupChildrenIdsToDelete;

    @Override
    public int compareTo(@Nonnull Group other) {
        return group.compareTo(other);
    }

    @Override
    public long getDirectoryId() {
        return group.getDirectoryId();
    }

    @Override
    public String getName() {
        return group.getName();
    }

    @Override
    public GroupType getType() {
        return group.getType();
    }

    @Override
    public boolean isActive() {
        return group.isActive();
    }

    @Override
    public String getDescription() {
        return group.getDescription();
    }

    @Nullable
    @Override
    public String getExternalId() {
        return group.getExternalId();
    }

    public GroupWithMembershipChanges merge(GroupWithMembershipChanges other) {
        return toBuilder()
                .addUserChildrenIdsToAdd(other.getUserChildrenIdsToAdd())
                .addUserChildrenIdsToDelete(other.getUserChildrenIdsToDelete())
                .addGroupChildrenIdsToAdd(other.getGroupChildrenIdsToAdd())
                .addGroupChildrenIdsToDelete(other.getGroupChildrenIdsToDelete())
                .build();
    }

    public Builder toBuilder() {
        return builder(this);
    }

    public static Builder builder(Group group) {
        return builder().setGroup(ImmutableGroup.from(group));
    }

// The code below has been generated by BoB the Builder of Beans based on the class' fields.
// Everything after this comment will be regenerated if you invoke BoB again.
// If you don't know who BoB is, you can find him here: https://bitbucket.org/atlassianlabs/bob-the-builder-of-beans

    protected GroupWithMembershipChanges(ImmutableGroup group, Iterable<String> userChildrenIdsToAdd, Iterable<String> userChildrenIdsToDelete, Iterable<String> groupChildrenIdsToAdd, Iterable<String> groupChildrenIdsToDelete) {
        this.group = Objects.requireNonNull(group);
        this.userChildrenIdsToAdd = ImmutableSet.copyOf(userChildrenIdsToAdd);
        this.userChildrenIdsToDelete = ImmutableSet.copyOf(userChildrenIdsToDelete);
        this.groupChildrenIdsToAdd = ImmutableSet.copyOf(groupChildrenIdsToAdd);
        this.groupChildrenIdsToDelete = ImmutableSet.copyOf(groupChildrenIdsToDelete);
    }

    public ImmutableGroup getGroup() {
        return group;
    }

    public Set<String> getUserChildrenIdsToAdd() {
        return userChildrenIdsToAdd;
    }

    public Set<String> getUserChildrenIdsToDelete() {
        return userChildrenIdsToDelete;
    }

    public Set<String> getGroupChildrenIdsToAdd() {
        return groupChildrenIdsToAdd;
    }

    public Set<String> getGroupChildrenIdsToDelete() {
        return groupChildrenIdsToDelete;
    }

    public static GroupWithMembershipChanges.Builder builder() {
        return new GroupWithMembershipChanges.Builder();
    }

    public static GroupWithMembershipChanges.Builder builder(GroupWithMembershipChanges data) {
        return new GroupWithMembershipChanges.Builder(data);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        GroupWithMembershipChanges that = (GroupWithMembershipChanges) o;

        return Objects.equals(this.getGroup(), that.getGroup()) && Objects.equals(this.getUserChildrenIdsToAdd(), that.getUserChildrenIdsToAdd()) && Objects.equals(this.getUserChildrenIdsToDelete(), that.getUserChildrenIdsToDelete()) && Objects.equals(this.getGroupChildrenIdsToAdd(), that.getGroupChildrenIdsToAdd()) && Objects.equals(this.getGroupChildrenIdsToDelete(), that.getGroupChildrenIdsToDelete());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getGroup(), getUserChildrenIdsToAdd(), getUserChildrenIdsToDelete(), getGroupChildrenIdsToAdd(), getGroupChildrenIdsToDelete());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("group", getGroup())
                .add("userChildrenIdsToAdd", getUserChildrenIdsToAdd())
                .add("userChildrenIdsToDelete", getUserChildrenIdsToDelete())
                .add("groupChildrenIdsToAdd", getGroupChildrenIdsToAdd())
                .add("groupChildrenIdsToDelete", getGroupChildrenIdsToDelete())
                .toString();
    }

    public static final class Builder {

        private ImmutableGroup group;
        private Set<String> userChildrenIdsToAdd = new HashSet<>();
        private Set<String> userChildrenIdsToDelete = new HashSet<>();
        private Set<String> groupChildrenIdsToAdd = new HashSet<>();
        private Set<String> groupChildrenIdsToDelete = new HashSet<>();

        private Builder() {
        }

        private Builder(GroupWithMembershipChanges initialData) {
            this.group = initialData.getGroup();
            this.userChildrenIdsToAdd = new HashSet<>(initialData.getUserChildrenIdsToAdd());
            this.userChildrenIdsToDelete = new HashSet<>(initialData.getUserChildrenIdsToDelete());
            this.groupChildrenIdsToAdd = new HashSet<>(initialData.getGroupChildrenIdsToAdd());
            this.groupChildrenIdsToDelete = new HashSet<>(initialData.getGroupChildrenIdsToDelete());
        }

        public Builder setGroup(ImmutableGroup group) {
            this.group = group;
            return this;
        }

        public Builder setUserChildrenIdsToAdd(Set<String> userChildrenIdsToAdd) {
            this.userChildrenIdsToAdd = userChildrenIdsToAdd;
            return this;
        }

        public Builder addUserChildrenIdsToAddItem(String userChildrenIdsToAddItem) {
            this.userChildrenIdsToAdd.add(userChildrenIdsToAddItem);
            return this;
        }

        public Builder addUserChildrenIdsToAdd(Iterable<String> userChildrenIdsToAdd) {
            for (String userChildrenIdsToAddItem : userChildrenIdsToAdd) {
                addUserChildrenIdsToAddItem(userChildrenIdsToAddItem);
            }
            return this;
        }

        public Builder setUserChildrenIdsToDelete(Set<String> userChildrenIdsToDelete) {
            this.userChildrenIdsToDelete = userChildrenIdsToDelete;
            return this;
        }

        public Builder addUserChildrenIdsToDeleteItem(String userChildrenIdsToDeleteItem) {
            this.userChildrenIdsToDelete.add(userChildrenIdsToDeleteItem);
            return this;
        }

        public Builder addUserChildrenIdsToDelete(Iterable<String> userChildrenIdsToDelete) {
            for (String userChildrenIdsToDeleteItem : userChildrenIdsToDelete) {
                addUserChildrenIdsToDeleteItem(userChildrenIdsToDeleteItem);
            }
            return this;
        }

        public Builder setGroupChildrenIdsToAdd(Set<String> groupChildrenIdsToAdd) {
            this.groupChildrenIdsToAdd = groupChildrenIdsToAdd;
            return this;
        }

        public Builder addGroupChildrenIdsToAddItem(String groupChildrenIdsToAddItem) {
            this.groupChildrenIdsToAdd.add(groupChildrenIdsToAddItem);
            return this;
        }

        public Builder addGroupChildrenIdsToAdd(Iterable<String> groupChildrenIdsToAdd) {
            for (String groupChildrenIdsToAddItem : groupChildrenIdsToAdd) {
                addGroupChildrenIdsToAddItem(groupChildrenIdsToAddItem);
            }
            return this;
        }

        public Builder setGroupChildrenIdsToDelete(Set<String> groupChildrenIdsToDelete) {
            this.groupChildrenIdsToDelete = groupChildrenIdsToDelete;
            return this;
        }

        public Builder addGroupChildrenIdsToDeleteItem(String groupChildrenIdsToDeleteItem) {
            this.groupChildrenIdsToDelete.add(groupChildrenIdsToDeleteItem);
            return this;
        }

        public Builder addGroupChildrenIdsToDelete(Iterable<String> groupChildrenIdsToDelete) {
            for (String groupChildrenIdsToDeleteItem : groupChildrenIdsToDelete) {
                addGroupChildrenIdsToDeleteItem(groupChildrenIdsToDeleteItem);
            }
            return this;
        }

        public GroupWithMembershipChanges build() {
            return new GroupWithMembershipChanges(group, userChildrenIdsToAdd, userChildrenIdsToDelete, groupChildrenIdsToAdd, groupChildrenIdsToDelete);
        }
    }
}
