package com.atlassian.crowd.directory.rest.resolver;

import com.atlassian.crowd.directory.rest.entity.membership.DirectoryObject;
import com.atlassian.crowd.directory.rest.entity.membership.GraphMembershipGroup;
import com.atlassian.crowd.directory.rest.entity.membership.GraphMembershipUser;
import com.google.common.collect.ImmutableMap;
import org.codehaus.jackson.annotate.JsonTypeInfo;
import org.codehaus.jackson.map.jsontype.TypeIdResolver;
import org.codehaus.jackson.map.type.TypeFactory;
import org.codehaus.jackson.map.util.ClassUtil;
import org.codehaus.jackson.type.JavaType;

import java.util.Map;

/**
 * TypeIdResolver for {@link DirectoryObject} instances. Resolves a concrete type basing on the @odata.type property
 * or returns a generic DirectoryObject type if the annotation is not present or its value does not indicate a concrete
 * type used in Crowd
 */
public class DirectoryObjectTypeIdResolver implements TypeIdResolver {

    public static final String USER_ODATA_TYPE = "#microsoft.graph.user";
    public static final String GROUP_ODATA_TYPE = "#microsoft.graph.group";
    private static final Map<Class<?>, String> CLASSES_TO_IDS = ImmutableMap.of(
            GraphMembershipUser.class, USER_ODATA_TYPE,
            GraphMembershipGroup.class, GROUP_ODATA_TYPE
    );

    private Map<String, JavaType> idsToTypes;
    private JavaType baseType;
    private JavaType fallbackType;

    @Override
    public void init(JavaType baseType) {
        this.baseType = baseType;
        idsToTypes = new ImmutableMap.Builder<String, JavaType>()
                .put(USER_ODATA_TYPE, constructJavaType(GraphMembershipUser.class))
                .put(GROUP_ODATA_TYPE, constructJavaType(GraphMembershipGroup.class))
                .build();
        fallbackType = constructJavaType(DirectoryObject.class);
    }

    @Override
    public String idFromValue(Object value) {
        return idFromValueAndType(value, value.getClass());
    }

    @Override
    public String idFromValueAndType(Object value, Class<?> suggestedType) {
        return CLASSES_TO_IDS.get(suggestedType);
    }

    @Override
    public JavaType typeFromId(String id) {
        return idsToTypes.getOrDefault(id, fallbackType);
    }

    private JavaType constructJavaType(Class<?> clazz) {
        try {
            final Class<?> classToCreate = ClassUtil.findClass(clazz.getName());
            return TypeFactory.defaultInstance().constructSpecializedType(baseType, classToCreate);
        } catch (ClassNotFoundException e) {
            throw new IllegalStateException("Cannot find class " + clazz);
        }
    }

    @Override
    public JsonTypeInfo.Id getMechanism() {
        return JsonTypeInfo.Id.CUSTOM;
    }
}
