package com.atlassian.crowd.directory.rest.mapper;

import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

/**
 * Encapsulates the result of a delta query. Contains the added/edited entities, names of deleted entities and a
 * synchronisation token to use for future incremental synchronisations
 * @param <T> The type used for added/changed entities
 */
public class DeltaQueryResult<T> {
    private static final DeltaQueryResult EMPTY = builder().build();

    private final List<T> changedEntities;
    /**
     * Contains the ids of entities that were reported as deleted by Azure AD. Entities reported as deleted have an
     * "@removed" attribute.
     */
    private final Set<String> deletedEntities;
    /**
     * Contains the ids of entities that were returned with an empty name (displayName in case of groups,
     * userPrincipalName in case of users). An empty name is either null, empty or consisting entirely of whitespace.
     */
    private final Set<String> namelessEntities;
    @Nullable
    private final String syncToken;

    public Builder<T> toBuilder() {
        return builder(this);
    }

    public static <T> Builder<T> builder(String syncToken) {
        return DeltaQueryResult.<T>builder().setSyncToken(syncToken);
    }

    public static <T> DeltaQueryResult<T> empty() {
        return (DeltaQueryResult<T>) EMPTY;
    }

// The code below has been generated by BoB the Builder of Beans based on the class' fields.
// Everything after this comment will be regenerated if you invoke BoB again.
// If you don't know who BoB is, you can find him here: https://bitbucket.org/atlassianlabs/bob-the-builder-of-beans

    protected DeltaQueryResult(Iterable<T> changedEntities, Iterable<String> deletedEntities, Iterable<String> namelessEntities, @Nullable String syncToken) {
        this.changedEntities = ImmutableList.copyOf(changedEntities);
        this.deletedEntities = ImmutableSet.copyOf(deletedEntities);
        this.namelessEntities = ImmutableSet.copyOf(namelessEntities);
        this.syncToken = syncToken;
    }

    public List<T> getChangedEntities() {
        return changedEntities;
    }

    public Set<String> getDeletedEntities() {
        return deletedEntities;
    }

    public Set<String> getNamelessEntities() {
        return namelessEntities;
    }

    public Optional<String> getSyncToken() {
        return Optional.ofNullable(syncToken);
    }

    public static <T> DeltaQueryResult.Builder<T> builder() {
        return new DeltaQueryResult.Builder<>();
    }

    public static <T> DeltaQueryResult.Builder<T> builder(DeltaQueryResult<T> data) {
        return new DeltaQueryResult.Builder<>(data);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        DeltaQueryResult that = (DeltaQueryResult) o;

        return Objects.equals(this.getChangedEntities(), that.getChangedEntities()) && Objects.equals(this.getDeletedEntities(), that.getDeletedEntities()) && Objects.equals(this.getNamelessEntities(), that.getNamelessEntities()) && Objects.equals(this.getSyncToken(), that.getSyncToken());
    }

    @Override
    public int hashCode() {
        return Objects.hash(getChangedEntities(), getDeletedEntities(), getNamelessEntities(), getSyncToken());
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("changedEntities", getChangedEntities())
                .add("deletedEntities", getDeletedEntities())
                .add("namelessEntities", getNamelessEntities())
                .add("syncToken", getSyncToken())
                .toString();
    }

    public static final class Builder<T> {

        private List<T> changedEntities = new ArrayList<>();
        private Set<String> deletedEntities = new HashSet<>();
        private Set<String> namelessEntities = new HashSet<>();
        private String syncToken;

        private Builder() {
        }

        private Builder(DeltaQueryResult<T> initialData) {
            this.changedEntities = new ArrayList<>(initialData.getChangedEntities());
            this.deletedEntities = new HashSet<>(initialData.getDeletedEntities());
            this.namelessEntities = new HashSet<>(initialData.getNamelessEntities());
            this.syncToken = initialData.getSyncToken().orElse(null);
        }

        public Builder<T> setChangedEntities(List<T> changedEntities) {
            this.changedEntities = changedEntities;
            return this;
        }

        public Builder<T> addChangedEntity(T changedEntity) {
            this.changedEntities.add(changedEntity);
            return this;
        }

        public Builder<T> addChangedEntities(Iterable<T> changedEntities) {
            for (T changedEntity : changedEntities) {
                addChangedEntity(changedEntity);
            }
            return this;
        }

        public Builder<T> setDeletedEntities(Set<String> deletedEntities) {
            this.deletedEntities = deletedEntities;
            return this;
        }

        public Builder<T> addDeletedEntity(String deletedEntity) {
            this.deletedEntities.add(deletedEntity);
            return this;
        }

        public Builder<T> addDeletedEntities(Iterable<String> deletedEntities) {
            for (String deletedEntity : deletedEntities) {
                addDeletedEntity(deletedEntity);
            }
            return this;
        }

        public Builder<T> setNamelessEntities(Set<String> namelessEntities) {
            this.namelessEntities = namelessEntities;
            return this;
        }

        public Builder<T> addNamelessEntity(String namelessEntity) {
            this.namelessEntities.add(namelessEntity);
            return this;
        }

        public Builder<T> addNamelessEntities(Iterable<String> namelessEntities) {
            for (String namelessEntity : namelessEntities) {
                addNamelessEntity(namelessEntity);
            }
            return this;
        }

        public Builder<T> setSyncToken(@Nullable String syncToken) {
            this.syncToken = syncToken;
            return this;
        }

        public DeltaQueryResult<T> build() {
            return new DeltaQueryResult<>(changedEntities, deletedEntities, namelessEntities, syncToken);
        }
    }
}
