package com.atlassian.crowd.directory.rest.mapper;

import com.atlassian.crowd.directory.query.MicrosoftGraphQueryTranslator;
import com.atlassian.crowd.directory.rest.delta.GraphDeltaQueryResult;
import com.atlassian.crowd.directory.rest.entity.GraphDirectoryObjectList;
import com.atlassian.crowd.directory.rest.entity.delta.GraphDeltaQueryGroup;
import com.atlassian.crowd.directory.rest.entity.delta.GraphDeltaQueryMembership;
import com.atlassian.crowd.directory.rest.entity.delta.GraphDeltaQueryUser;
import com.atlassian.crowd.directory.rest.entity.group.GraphGroup;
import com.atlassian.crowd.directory.rest.entity.group.GraphGroupList;
import com.atlassian.crowd.directory.rest.entity.membership.DirectoryObject;
import com.atlassian.crowd.directory.rest.entity.membership.GraphMembershipGroup;
import com.atlassian.crowd.directory.rest.entity.membership.GraphMembershipUser;
import com.atlassian.crowd.directory.rest.entity.user.GraphUser;
import com.atlassian.crowd.directory.rest.entity.user.GraphUsersList;
import com.atlassian.crowd.directory.rest.resolver.DirectoryObjectTypeIdResolver;
import com.atlassian.crowd.model.group.Group;
import com.atlassian.crowd.model.group.GroupTemplate;
import com.atlassian.crowd.model.group.GroupTemplateWithAttributes;
import com.atlassian.crowd.model.group.GroupWithMembershipChanges;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.model.user.UserTemplate;
import com.atlassian.crowd.model.user.UserTemplateWithAttributes;
import com.atlassian.crowd.model.user.UserWithAttributes;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableTable;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.util.UriComponentsBuilder;

import javax.annotation.Nonnull;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Maps REST entities returned from Microsoft Graph to Crowd entities and vice versa
 */
public class AzureAdRestEntityMapper {

    private static final Logger log = LoggerFactory.getLogger(AzureAdRestEntityMapper.class);

    /**
     * Maps a users response from Azure AD to Crowd users.
     *
     * @param graphUsersList the response from Azure AD
     * @param returnType     the desired return type
     * @param directoryId    the id of the Crowd directory
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @return a list of mapped Crowd users
     */
    public <T> List<T> mapUsers(final GraphUsersList graphUsersList, Class<T> returnType, final long directoryId, String alternativeUsernameAttribute) {
        return mapUsers(graphUsersList.getEntries(), returnType, directoryId, alternativeUsernameAttribute);
    }

    /**
     * Maps a collection of Azure AD users to Crowd users.
     *
     * @param graphUsersList the users to map
     * @param returnType     the desired return type
     * @param directoryId    the id of the Crowd directory
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @return a list of mapped Crowd users
     */
    public <T> List<T> mapUsers(final Collection<GraphUser> graphUsersList, Class<T> returnType, final long directoryId, String alternativeUsernameAttribute) {
        return graphUsersList.stream().map(graphUser -> mapUser(graphUser, returnType, directoryId, alternativeUsernameAttribute)).collect(Collectors.toList());
    }

    /**
     * Maps a delta query result for users to a result suitable for incremental synchronisation.
     *
     * @param graphUsersList the users to map
     * @param directoryId    the id of the Crowd directory
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @return a list of mapped new users, ids of deleted users and a token suitable for change tracking
     */
    public DeltaQueryResult<UserWithAttributes> mapDeltaQueryUsers(final GraphDeltaQueryResult<GraphDeltaQueryUser> graphUsersList,
                                                                   final long directoryId,
                                                                   final String alternativeUsernameAttribute) {
        final DeltaQueryResult.Builder<UserWithAttributes> builder =
                DeltaQueryResult.builder(getDeltaToken(graphUsersList.getDeltaLink()));
        graphUsersList.getResults().forEach(graphUser -> {
            if (graphUser.getRemoved() == null) {
                final String username = getUsername(graphUser, alternativeUsernameAttribute);
                if (StringUtils.isNotBlank(username)) {
                    builder.addChangedEntity(mapUser(graphUser, UserWithAttributes.class, directoryId, alternativeUsernameAttribute));
                } else {
                    log.debug("Encountered a nameless user from Azure AD, name: {}, id: {}",
                            username, graphUser.getId());
                    builder.addNamelessEntity(graphUser.getId());
                }
            } else {
                builder.addDeletedEntity(graphUser.getId());
            }
        });
        return buildAndlogDeltaQueryResults("users", builder);
    }

    private <T> DeltaQueryResult<T> buildAndlogDeltaQueryResults(String entityType, DeltaQueryResult.Builder<T> builder) {
        DeltaQueryResult<T> result = builder.build();
        log.debug("Mapped delta query {} - Changed: {}, removed: {}, nameless: {}, delta token: {}",
                entityType,
                result.getChangedEntities(),
                result.getDeletedEntities(),
                result.getNamelessEntities(),
                result.getSyncToken()
        );
        return result;
    }

    /**
     * Maps a user obtained from the Azure AD users endpoint to a Crowd object
     *
     * @param graphUser   the user obtained from Azure AD
     * @param returnType  the desired return type
     * @param directoryId the id of the Crowd directory
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @return the mapped Crowd user
     */
    public <T> T mapUser(final GraphUser graphUser, Class<T> returnType, final long directoryId, String alternativeUsernameAttribute) {
        final String username = getUsername(graphUser, alternativeUsernameAttribute);
        if (returnType == String.class) {
            return (T) username;
        } else {
            final UserTemplate userTemplate = new UserTemplate(username, graphUser.getGivenName(),
                    graphUser.getSurname(), graphUser.getDisplayName());
            userTemplate.setDirectoryId(directoryId);
            userTemplate.setEmailAddress(graphUser.getMail()); // TODO: mail property seems wonky, consider using the user's username and handling EXT if empty
            final boolean mappedActive = graphUser.getAccountEnabled() != null ? graphUser.getAccountEnabled() : true;
            log.trace("Mapped active flag for user {} from {} to {}", graphUser.getUserPrincipalName(), graphUser.getAccountEnabled(), mappedActive);
            userTemplate.setActive(mappedActive);
            userTemplate.setExternalId(graphUser.getId());
            log.debug("Mapped Graph user to Crowd user {}", userTemplate);
            if (returnType == User.class) {
                return (T) userTemplate;
            } else {
                return (T) UserTemplateWithAttributes.toUserWithNoAttributes(userTemplate);
            }
        }
    }

    /**
     * Maps a groups response from Azure AD to Crowd groups.
     *
     * @param graphGroupList the response from Azure AD
     * @param returnType     the desired return type
     * @param directoryId    the id of the Crowd directory
     * @return a list of mapped Crowd groups
     */
    public <T> List<T> mapGroups(final GraphGroupList graphGroupList, Class<T> returnType, final long directoryId) {
        return mapGroups(graphGroupList.getEntries(), returnType, directoryId);
    }

    /**
     * Maps a collection of Azure AD groups to Crowd groups.
     *
     * @param graphGroups the groups to map
     * @param returnType  the desired return type
     * @param directoryId the id of the Crowd directory
     * @return a list of mapped Crowd groups
     */
    public <T> List<T> mapGroups(final Collection<GraphGroup> graphGroups, Class<T> returnType, final long directoryId) {
        return graphGroups.stream().map(graphGroup -> mapGroup(graphGroup, returnType, directoryId)).collect(Collectors.toList());
    }

    /**
     * Maps a delta query result for groups to a result suitable for incremental synchronisation.
     *
     * @param graphGroups the groups to map
     * @param directoryId the id of the Crowd directory
     * @return a list of mapped new groups, ids of deleted groups and a token suitable for change tracking
     */
    public DeltaQueryResult<GroupWithMembershipChanges> mapDeltaQueryGroups(final GraphDeltaQueryResult<GraphDeltaQueryGroup> graphGroups, final long directoryId) {
        final Map<String, GroupWithMembershipChanges> changedGroups = new HashMap<>();
        DeltaQueryResult.Builder<GroupWithMembershipChanges> builder =
                DeltaQueryResult.builder(getDeltaToken(graphGroups.getDeltaLink()));
        graphGroups.getResults().forEach(graphGroup -> {
            if (graphGroup.getRemoved() == null) {
                if (StringUtils.isNotBlank(graphGroup.getDisplayName())) {
                    changedGroups.merge(graphGroup.getId(), mapDeltaQueryGroup(graphGroup, directoryId), GroupWithMembershipChanges::merge);
                } else {
                    log.debug("Encountered a nameless group from Azure AD, name: {}, id: {}",
                            graphGroup.getDisplayName(), graphGroup.getId());
                    builder.addNamelessEntity(graphGroup.getId());
                }
            } else {
                builder.addDeletedEntity(graphGroup.getId());
            }
        });
        return buildAndlogDeltaQueryResults("groups", builder.addChangedEntities(changedGroups.values()));
    }

    private static String getDeltaToken(String deltaLink) {
        final List<String> deltatoken = UriComponentsBuilder.fromUriString(deltaLink).build().getQueryParams().get("$deltatoken");
        return Iterables.getOnlyElement(deltatoken);
    }

    /**
     * Maps a directory objects response from Azure AD. The objects in the response should be of one type (groups or users)
     *
     * @param directoryObjectList the directory objects response from Azure AD
     * @param returnType          the desired return type
     * @param directoryId         the id of the Crowd directory
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @return a list of mapped Crowd objects
     */
    public <T> List<T> mapDirectoryObjects(final GraphDirectoryObjectList directoryObjectList,
                                           Class<T> returnType,
                                           final long directoryId,
                                           String alternativeUsernameAttribute) {
        return mapDirectoryObjects(directoryObjectList.getEntries(), returnType, directoryId, alternativeUsernameAttribute);
    }

    /**
     * Maps a collection of directory objects to Crowd objects. The objects should be filtered by type (groups or users)
     *
     * @param directoryObjects the directory objects to map
     * @param returnType       the desired return type
     * @param directoryId      the id of the Crowd directory
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @return a list of mapped Crowd objects
     */
    public <T> List<T> mapDirectoryObjects(final Collection<DirectoryObject> directoryObjects,
                                           Class<T> returnType,
                                           final long directoryId,
                                           String alternativeUsernameAttribute) {
        return directoryObjects.stream()
                .map(graphGroup -> mapDirectoryObject(graphGroup, returnType, directoryId, alternativeUsernameAttribute))
                .collect(Collectors.toList());
    }

    /**
     * Maps a group obtained from the Azure AD groups endpoint to a Crowd object
     *
     * @param graphGroup  the group obtained from Azure AD
     * @param returnType  the desired return type
     * @param directoryId the id of the Crowd directory
     * @return the mapped Crowd group
     */
    public <T> T mapGroup(final GraphGroup graphGroup, Class<T> returnType, final long directoryId) {
        if (returnType == String.class) {
            return (T) graphGroup.getDisplayName();
        } else {
            final GroupTemplate groupTemplate = new GroupTemplate(graphGroup.getDisplayName(), directoryId);
            groupTemplate.setDescription(graphGroup.getDescription());
            groupTemplate.setExternalId(graphGroup.getId());
            log.debug("Mapped Graph group to Crowd group {}", groupTemplate);
            if (returnType == Group.class) {
                return (T) groupTemplate;
            } else {
                return (T) GroupTemplateWithAttributes.ofGroupWithNoAttributes(groupTemplate);
            }
        }
    }

    /**
     * Maps a group obtained from the Azure AD groups endpoint to a Crowd object
     *
     * @param graphGroup  the group obtained from Azure AD
     * @param directoryId the id of the Crowd directory
     * @return the mapped Crowd group
     */
    public GroupWithMembershipChanges mapDeltaQueryGroup(final GraphDeltaQueryGroup graphGroup, final long directoryId) {
        final GroupTemplate groupTemplate = new GroupTemplate(graphGroup.getDisplayName(), directoryId);
        groupTemplate.setDescription(graphGroup.getDescription());
        groupTemplate.setExternalId(graphGroup.getId());
        GroupWithMembershipChanges.Builder builder = GroupWithMembershipChanges.builder(groupTemplate);
        for (GraphDeltaQueryMembership membership : graphGroup.getMembers()) {
            Optional.ofNullable(BUILDER_FUNCTION_MAP.get(membership.getType(), membership.getRemoved() == null))
                .ifPresent(f -> f.accept(builder, membership.getId()));
        }
        return builder.build();
    }

    private static final ImmutableTable<String, Boolean, BiConsumer<GroupWithMembershipChanges.Builder, String>> BUILDER_FUNCTION_MAP =
            ImmutableTable.<String, Boolean, BiConsumer<GroupWithMembershipChanges.Builder, String>>builder()
                    .put(DirectoryObjectTypeIdResolver.USER_ODATA_TYPE, true, GroupWithMembershipChanges.Builder::addUserChildrenIdsToAddItem)
                    .put(DirectoryObjectTypeIdResolver.USER_ODATA_TYPE, false, GroupWithMembershipChanges.Builder::addUserChildrenIdsToDeleteItem)
                    .put(DirectoryObjectTypeIdResolver.GROUP_ODATA_TYPE, true, GroupWithMembershipChanges.Builder::addGroupChildrenIdsToAddItem)
                    .put(DirectoryObjectTypeIdResolver.GROUP_ODATA_TYPE, false, GroupWithMembershipChanges.Builder::addGroupChildrenIdsToDeleteItem)
                    .build();

    /**
     * Maps an AzureAD directory object to a Crowd object. The directory object must be of a concrete type
     *
     * @param directoryObject the result from Azure AD
     * @param returnType      the desired return type
     * @param directoryId     the id of the Crowd directory
     * @return the mapped Crowd object
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     * @throws IllegalArgumentException when the directoryObject is a generic {@link DirectoryObject} instance
     */
    public <T> T mapDirectoryObject(final DirectoryObject directoryObject, Class<T> returnType, final long directoryId, String alternativeUsernameAttribute) {
        if (directoryObject instanceof GraphMembershipGroup) {
            if (returnType == String.class) {
                return (T) directoryObject.getDisplayName();
            } else {
                final GroupTemplate groupTemplate = new GroupTemplate(directoryObject.getDisplayName(), directoryId);
                groupTemplate.setDescription(((GraphMembershipGroup) directoryObject).getDescription());
                groupTemplate.setExternalId(directoryObject.getId());
                log.debug("Mapped Graph group to Crowd group {}", groupTemplate);
                if (returnType == Group.class) {
                    return (T) groupTemplate;
                } else {
                    return (T) GroupTemplateWithAttributes.ofGroupWithNoAttributes(groupTemplate);
                }
            }
        } else if (directoryObject instanceof GraphMembershipUser) {
            GraphMembershipUser graphUser = (GraphMembershipUser) directoryObject;
            final String username = getUsername(graphUser, alternativeUsernameAttribute);
            if (returnType == String.class) {
                return (T) username;
            } else {
                final UserTemplate userTemplate = new UserTemplate(username, graphUser.getGivenName(),
                        graphUser.getSurname(), graphUser.getDisplayName());
                userTemplate.setDirectoryId(directoryId);
                userTemplate.setEmailAddress(graphUser.getMail());
                userTemplate.setActive(MoreObjects.firstNonNull(graphUser.getAccountEnabled(), true));
                userTemplate.setExternalId(directoryObject.getId());
                log.debug("Mapped Graph user to Crowd user {}", userTemplate);
                if (returnType == User.class) {
                    return (T) userTemplate;
                } else {
                    return (T) UserTemplateWithAttributes.toUserWithNoAttributes(userTemplate);
                }
            }
        } else {
            throw new IllegalArgumentException("Cannot map directory object of type " + directoryObject.getClass());
        }
    }

    /**
     * Returns name of the user.
     * @param user {@link GraphUser} returned by Azure
     * @param alternativeUsernameAttribute name of the attribute that should be mapped to user name
     */
    public String getUsername(GraphUser user, String alternativeUsernameAttribute) {
        return getUsername(user.getUserPrincipalName(), getAlternateUsername(user, GRAPH_USER_GETTERS, alternativeUsernameAttribute));
    }

    private String getUsername(GraphMembershipUser user, String alternativeUsernameAttribute) {
        return getUsername(user.getUserPrincipalName(), getAlternateUsername(user, GRAPH_MEMBERSHIP_USER_GETTERS, alternativeUsernameAttribute));
    }

    private <T> String getAlternateUsername(T user, Map<String, Function<T, String>> getters, String attributeName) {
        if (StringUtils.isEmpty(attributeName) || MicrosoftGraphQueryTranslator.USERNAME.equals(attributeName)) {
            return null;
        }
        Function<T, String> getter = getters.get(attributeName);
        Preconditions.checkArgument(getter != null, "'%s' is not a valid username attribute", attributeName);
        return getter.apply(user);
    }

    private static final Map<String, Function<GraphMembershipUser, String>> GRAPH_MEMBERSHIP_USER_GETTERS = ImmutableMap.of(
            MicrosoftGraphQueryTranslator.MAIL, GraphMembershipUser::getMail,
            MicrosoftGraphQueryTranslator.DISPLAY_NAME, GraphMembershipUser::getDisplayName,
            MicrosoftGraphQueryTranslator.FIRST_NAME, GraphMembershipUser::getGivenName,
            MicrosoftGraphQueryTranslator.LAST_NAME, GraphMembershipUser::getSurname);

    private static final Map<String, Function<GraphUser, String>> GRAPH_USER_GETTERS = ImmutableMap.of(
            MicrosoftGraphQueryTranslator.MAIL, GraphUser::getMail,
            MicrosoftGraphQueryTranslator.DISPLAY_NAME, GraphUser::getDisplayName,
            MicrosoftGraphQueryTranslator.FIRST_NAME, GraphUser::getGivenName,
            MicrosoftGraphQueryTranslator.LAST_NAME, GraphUser::getSurname);

    private static final String EXTERNAL_UPN_FRAGMENT = "#EXT#";
    private static final String DOMAIN_START = "@";

    private <T> String getUsername(String upn, String alternateUsername) {
        if (isExternalUpn(upn) && StringUtils.isNotEmpty(alternateUsername) && !sameDomain(upn, alternateUsername)) {
            return alternateUsername;
        }
        return upn;
    }

    private boolean sameDomain(@Nonnull String upn, @Nonnull String alternateUsername) {
        // Do not allow falling back to usernames within the same tenant - this may lead to duplicates.
        int upnDomainStartPos = upn.lastIndexOf(DOMAIN_START);
        return upnDomainStartPos >= 0 && alternateUsername.endsWith(upn.substring(upnDomainStartPos));
    }

    private boolean isExternalUpn(String upn) {
        return StringUtils.contains(upn, EXTERNAL_UPN_FRAGMENT);
    }
}
