package com.atlassian.crowd.directory.rest;

import com.atlassian.crowd.directory.query.ODataTop;
import com.atlassian.crowd.directory.rest.delta.GraphDeltaQueryResult;
import com.atlassian.crowd.directory.rest.entity.PageableGraphList;
import com.atlassian.crowd.directory.rest.entity.delta.PageableDeltaQueryGraphList;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.search.query.entity.EntityQuery;
import com.atlassian.crowd.search.util.OrderedResultsConstrainer;
import com.google.common.collect.Iterables;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

/**
 * Wrapper class to facilitate paging results from Microsoft Graph
 */
public class AzureAdPagingWrapper {

    private final AzureAdRestClient azureAdRestClient;

    public AzureAdPagingWrapper(final AzureAdRestClient azureAdRestClient) {
        this.azureAdRestClient = azureAdRestClient;
    }

    /**
     * Will fetch results from Microsoft Graph until the desired amount of results is reached or until the results are exhausted
     *
     * @param firstPage    The first page of the results
     * @param maxResults   the amount of results that should be fetched in the same format as used by {@link EntityQuery}
     * @param <V>          Class representing a page of results
     * @param <T>          Class representing the entries contained in a page
     * @return list of search result entries of size up to maxResults
     */
    public <V extends PageableGraphList<T>, T> List<T> fetchAppropriateAmountOfResults(final V firstPage, final int startIndex, final int maxResults) throws OperationFailedException {
        return fetchResults(firstPage, null, startIndex, maxResults);
    }

    /**
     * Will fetch all results from Microsoft Graph that match the specified predicate
     * @param firstPage The first page of the results
     * @param filter the filter that results should match
     * @param <V> Class representing a page of results
     * @param <T> Class representing the entries contained in a page
     * @return list of search result entries matching the specified predicate
     */
    public <V extends PageableGraphList<T>, T> List<T> fetchAllMatchingResults(V firstPage, Predicate<T> filter) throws OperationFailedException {
        return fetchResults(firstPage, filter, 0, EntityQuery.ALL_RESULTS);
    }

    /**
     * Will fetch all results from Microsoft Graph
     * @param firstPage The first page of the results
     * @param <V> Class representing a page of results
     * @param <T> Class representing the entries contained in a page
     * @return list of search result entries matching the specified predicate
     */
    public <V extends PageableGraphList<T>, T> List<T> fetchAllResults(V firstPage) throws OperationFailedException {
        return fetchResults(firstPage, null, 0, EntityQuery.ALL_RESULTS);
    }

    /**
     * Retrieves the first element matching the specified predicate, if any, paging for it if necessary
     * @param firstPage The first page of the results
     * @param predicate The predicate against which search items will be matched
     * @param <V> Class representing a page of results
     * @param <T> Class representing the entries contained in a page
     * @return an Optional containing the matching element if found
     */
    public <V extends PageableGraphList<T>, T> Optional<T> pageForElement(final V firstPage, final Predicate<T> predicate) throws OperationFailedException {
        List<T> results = fetchResults(firstPage, predicate, 0, 1);
        return Optional.ofNullable(Iterables.getFirst(results, null));
    }

    public <V extends PageableDeltaQueryGraphList<T>, T> GraphDeltaQueryResult<T> fetchAllDeltaQueryResults(final V firstPage) throws OperationFailedException {
        Pair<List<T>, V> result = fetchResultsAndLastPage(firstPage, null, 0, EntityQuery.ALL_RESULTS);
        return new GraphDeltaQueryResult<>(result.getLeft(), result.getRight().getDeltaLink());
    }

    private  <V extends PageableGraphList<T>, T> List<T> fetchResults(
            final V firstPage, Predicate<T> filter, final int startIndex, final int maxResults) throws OperationFailedException {
        return fetchResultsAndLastPage(firstPage, filter, startIndex, maxResults).getLeft();
    }

    private <V extends PageableGraphList<T>, T> Pair<List<T>, V> fetchResultsAndLastPage(
            final V firstPage, Predicate<T> filter, final int startIndex, final int maxResults) throws OperationFailedException {
        boolean fullPageQuery = filter != null || maxResults == EntityQuery.ALL_RESULTS;
        OrderedResultsConstrainer<T> constrainer = new OrderedResultsConstrainer<>(filter, startIndex, maxResults);
        V currentPage = firstPage;
        while (true) {
            constrainer.addAll(currentPage.getEntries());
            if (constrainer.getRemainingCount() == 0 || StringUtils.isBlank(currentPage.getNextLink())) {
                break;
            }
            ODataTop top = fullPageQuery ? ODataTop.FULL_PAGE : ODataTop.forSize(constrainer.getRemainingCount());
            currentPage = azureAdRestClient.getNextPage(currentPage.getNextLink(), (Class<V>) currentPage.getClass(), top);
        }
        return Pair.of(constrainer.toList(), currentPage);
    }
}
