package com.atlassian.crowd.directory.query;

import java.util.Objects;

/**
 * Represents a $top query parameter used by Microsoft Graph. Used to set the the amount of results to return.
 * See: http://docs.oasis-open.org/odata/odata/v4.0/errata03/os/complete/part2-url-conventions/odata-v4.0-errata03-os-part2-url-conventions-complete.html
 */
public class ODataTop implements MicrosoftGraphQueryParam {

    /**
     * Maximum page size allowed by the Microsoft Graph API, see https://developer.microsoft.com/en-us/graph/docs/concepts/paging
     */
    public static final int MS_GRAPH_MAX_PAGE_SIZE = 999;
    public static final String QUERY_PARAM_NAME = "$top";
    public static final ODataTop FULL_PAGE = new ODataTop(MS_GRAPH_MAX_PAGE_SIZE);
    public static final ODataTop SINGLE_RESULT = new ODataTop(1);
    private final int value;

    public ODataTop(int value) {
        this.value = value;
    }

    @Override
    public String getName() {
        return QUERY_PARAM_NAME;
    }

    @Override
    public String asRawValue() {
        return String.valueOf(value);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ODataTop oDataTop = (ODataTop) o;
        return value == oDataTop.value;
    }

    public static ODataTop forSize(int size) {
        return size >= MS_GRAPH_MAX_PAGE_SIZE ? FULL_PAGE : new ODataTop(size);
    }

    @Override
    public int hashCode() {
        return Objects.hash(value);
    }
}
