package com.atlassian.crowd.directory.query;

import java.util.Objects;
import java.util.stream.Stream;

import static java.util.stream.Collectors.joining;

/**
 * Represents a $select query parameter used by Microsoft Graph. Used to specify the returned columns
 * See: http://docs.oasis-open.org/odata/odata/v4.0/errata03/os/complete/part2-url-conventions/odata-v4.0-errata03-os-part2-url-conventions-complete.html
 */
public class ODataSelect implements MicrosoftGraphQueryParam {
    private static final String SEPARATOR = ",";

    public static final String QUERY_PARAM_NAME = "$select";
    private final String value;

    public ODataSelect(final String... values) {
        this(Stream.of(values));
    }

    private ODataSelect(Stream<String> stream) {
        this.value = stream.distinct().collect(joining(SEPARATOR));
    }

    public ODataSelect addColumns(String... values) {
        return new ODataSelect(Stream.concat(Stream.of(value.split(SEPARATOR)), Stream.of(values)));
    }

    public ODataSelect merge(ODataSelect other) {
        return addColumns(other.value.split(SEPARATOR));
    }

    @Override
    public String getName() {
        return QUERY_PARAM_NAME;
    }

    public String asRawValue() {
        return value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ODataSelect that = (ODataSelect) o;
        return Objects.equals(value, that.value);
    }

    @Override
    public int hashCode() {
        return Objects.hash(value);
    }
}
